<?php
/**
 * API Handler
 * Handles all AJAX requests
 */

session_start();

// Include required files
require_once '../config/config.php';
require_once '../config/database.php';
require_once '../includes/functions.php';
require_once '../includes/ai-generator.php';

// Set content type
header('Content-Type: application/json');

// Check if user is logged in
if (!is_logged_in()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Authentication required']);
    exit;
}

// Get request data
$input = json_decode(file_get_contents('php://input'), true);
$action = $_GET['action'] ?? '';

// Rate limiting
if (!check_rate_limit(get_client_ip(), $action)) {
    http_response_code(429);
    echo json_encode(['success' => false, 'error' => 'Rate limit exceeded']);
    exit;
}

try {
    switch ($action) {
        case 'generate':
            handle_generation($input);
            break;
            
        case 'save_project':
            handle_save_project($input);
            break;
            
        case 'get_project':
            handle_get_project($input);
            break;
            
        case 'delete_project':
            handle_delete_project($input);
            break;
            
        case 'export_project':
            handle_export_project($input);
            break;
            
        case 'get_templates':
            handle_get_templates();
            break;
            
        case 'get_statistics':
            handle_get_statistics();
            break;
            
        default:
            throw new Exception('Invalid action');
    }
    
} catch (Exception $e) {
    log_error("API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'error' => $e->getMessage()
    ]);
}

/**
 * Handle app generation
 */
function handle_generation($data) {
    // Validate input
    if (empty($data['title']) || empty($data['prompt']) || empty($data['app_type'])) {
        throw new Exception('Missing required fields');
    }
    
    // Check CSRF token
    if (!verify_csrf_token($data['csrf_token'] ?? '')) {
        throw new Exception('Invalid security token');
    }
    
    $user_id = get_current_user_id();
    $title = sanitize_input($data['title']);
    $prompt = $data['prompt'];
    $app_type = sanitize_input($data['app_type']);
    $features = $data['features'] ?? [];
    
    // Create project record
    $project_id = create_project($user_id, $title, $prompt, $app_type);
    if (!$project_id) {
        throw new Exception('Failed to create project');
    }
    
    // Log generation request
    log_generation_request($user_id, $project_id, $prompt, 'start');
    
    // Generate the application using AI
    $ai_generator = new AIGenerator();
    $result = $ai_generator->generate_app($prompt, $app_type, 'html');
    
    if (!$result['success']) {
        update_project($project_id, ['status' => 'error']);
        throw new Exception($result['error'] ?? 'Generation failed');
    }
    
    // Save generated files
    $files_saved = save_project_files($project_id, $result['files']);
    if (!$files_saved) {
        update_project($project_id, ['status' => 'error']);
        throw new Exception('Failed to save project files');
    }
    
    // Update project status
    $update_data = [
        'description' => 'Generated using AI: ' . substr($prompt, 0, 200) . '...',
        'status' => 'complete',
        'generated_files' => json_encode(array_keys($result['files'])),
        'file_size' => calculate_total_size($result['files'])
    ];
    
    update_project($project_id, $update_data);
    
    // Log successful generation
    log_generation_request($user_id, $project_id, $prompt, 'success');
    
    echo json_encode([
        'success' => true,
        'project_id' => $project_id,
        'files' => array_keys($result['files']),
        'structure' => $result['structure']
    ]);
}

/**
 * Save project
 */
function handle_save_project($data) {
    if (empty($data['project_id'])) {
        throw new Exception('Project ID required');
    }
    
    $user_id = get_current_user_id();
    $project_id = $data['project_id'];
    
    // Check if project belongs to user
    $project = get_project_by_id($project_id, $user_id);
    if (!$project) {
        throw new Exception('Project not found');
    }
    
    // Update project
    $update_data = [];
    if (isset($data['title'])) $update_data['title'] = sanitize_input($data['title']);
    if (isset($data['description'])) $update_data['description'] = sanitize_input($data['description']);
    if (isset($data['status'])) $update_data['status'] = sanitize_input($data['status']);
    
    if (!empty($update_data)) {
        $update_data['updated_at'] = date('Y-m-d H:i:s');
        update_project($project_id, $update_data);
    }
    
    echo json_encode(['success' => true, 'message' => 'Project saved']);
}

/**
 * Get project
 */
function handle_get_project($data) {
    if (empty($data['project_id'])) {
        throw new Exception('Project ID required');
    }
    
    $user_id = get_current_user_id();
    $project_id = $data['project_id'];
    
    $project = get_project_by_id($project_id, $user_id);
    if (!$project) {
        throw new Exception('Project not found');
    }
    
    $files = get_project_files($project_id);
    
    echo json_encode([
        'success' => true,
        'project' => $project,
        'files' => $files
    ]);
}

/**
 * Delete project
 */
function handle_delete_project($data) {
    if (empty($data['project_id'])) {
        throw new Exception('Project ID required');
    }
    
    if (!verify_csrf_token($data['csrf_token'] ?? '')) {
        throw new Exception('Invalid security token');
    }
    
    $user_id = get_current_user_id();
    $project_id = $data['project_id'];
    
    $deleted = delete_project($project_id, $user_id);
    if (!$deleted) {
        throw new Exception('Failed to delete project');
    }
    
    echo json_encode(['success' => true, 'message' => 'Project deleted']);
}

/**
 * Export project
 */
function handle_export_project($data) {
    if (empty($data['project_id'])) {
        throw new Exception('Project ID required');
    }
    
    $user_id = get_current_user_id();
    $project_id = $data['project_id'];
    
    $project = get_project_by_id($project_id, $user_id);
    if (!$project) {
        throw new Exception('Project not found');
    }
    
    $files = get_project_files($project_id);
    if (empty($files)) {
        throw new Exception('No files found for this project');
    }
    
    // Create temporary zip file
    $zip_filename = 'project_' . $project_id . '_' . date('Y-m-d') . '.zip';
    $zip_path = sys_get_temp_dir() . '/' . $zip_filename;
    
    $zip = new ZipArchive();
    if ($zip->open($zip_path, ZipArchive::CREATE) !== TRUE) {
        throw new Exception('Failed to create zip file');
    }
    
    // Add files to zip
    foreach ($files as $file) {
        if (!empty($file['content'])) {
            $zip->addFromString($file['file_path'], $file['content']);
        } elseif (file_exists(PROJECTS_DIR . $project_id . '/' . $file['file_path'])) {
            $zip->addFile(PROJECTS_DIR . $project_id . '/' . $file['file_path'], $file['file_path']);
        }
    }
    
    $zip->close();
    
    // Log export
    global $db;
    $db->query("UPDATE projects SET view_count = view_count + 1 WHERE id = ?", [$project_id]);
    
    // Return file info for download
    echo json_encode([
        'success' => true,
        'download_url' => 'download.php?file=' . $zip_filename,
        'filename' => $zip_filename,
        'file_size' => filesize($zip_path)
    ]);
}

/**
 * Get templates
 */
function handle_get_templates() {
    $templates = get_app_templates();
    echo json_encode([
        'success' => true,
        'templates' => $templates
    ]);
}

/**
 * Get user statistics
 */
function handle_get_statistics() {
    $user_id = get_current_user_id();
    $stats = get_user_statistics($user_id);
    echo json_encode([
        'success' => true,
        'statistics' => $stats
    ]);
}

/**
 * Helper functions
 */
function log_generation_request($user_id, $project_id, $prompt, $status) {
    global $db;
    
    try {
        $stmt = $db->query(
            "INSERT INTO generation_logs (user_id, project_id, prompt, status, created_at) VALUES (?, ?, ?, ?, NOW())",
            [$user_id, $project_id, $prompt, $status]
        );
    } catch (Exception $e) {
        log_error("Failed to log generation: " . $e->getMessage());
    }
}

function save_project_files($project_id, $files) {
    global $db;
    
    try {
        $db->beginTransaction();
        
        // Create project directory
        $project_dir = PROJECTS_DIR . $project_id;
        create_directory($project_dir);
        
        // Save each file
        foreach ($files as $file_path => $content) {
            $file_name = basename($file_path);
            $full_path = $project_dir . '/' . $file_path;
            
            // Create directory for file if needed
            $file_dir = dirname($full_path);
            create_directory($file_dir);
            
            // Save file content
            file_put_contents($full_path, $content);
            
            // Save to database
            $file_size = strlen($content);
            $file_type = get_file_type($file_path);
            
            $db->query(
                "INSERT INTO project_files (project_id, file_path, file_name, file_type, file_size, content) VALUES (?, ?, ?, ?, ?, ?) 
                 ON DUPLICATE KEY UPDATE file_size = VALUES(file_size), content = VALUES(content)",
                [$project_id, $file_path, $file_name, $file_type, $file_size, $content]
            );
        }
        
        $db->commit();
        return true;
        
    } catch (Exception $e) {
        $db->rollBack();
        log_error("Failed to save project files: " . $e->getMessage());
        return false;
    }
}

function get_project_by_id($project_id, $user_id) {
    global $db;
    
    try {
        $stmt = $db->query("SELECT * FROM projects WHERE id = ? AND user_id = ?", [$project_id, $user_id]);
        return $stmt->fetch();
    } catch (Exception $e) {
        log_error("Failed to get project: " . $e->getMessage());
        return null;
    }
}

function get_project_files($project_id) {
    global $db;
    
    try {
        $stmt = $db->query("SELECT * FROM project_files WHERE project_id = ? ORDER BY file_path", [$project_id]);
        return $stmt->fetchAll();
    } catch (Exception $e) {
        log_error("Failed to get project files: " . $e->getMessage());
        return [];
    }
}

function calculate_total_size($files) {
    $total_size = 0;
    foreach ($files as $content) {
        $total_size += strlen($content);
    }
    return $total_size;
}

function get_file_type($file_path) {
    $ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
    
    $type_map = [
        'html' => 'text/html',
        'css' => 'text/css',
        'js' => 'application/javascript',
        'php' => 'application/php',
        'json' => 'application/json',
        'md' => 'text/markdown',
        'txt' => 'text/plain',
        'xml' => 'application/xml',
        'svg' => 'image/svg+xml',
        'png' => 'image/png',
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'gif' => 'image/gif'
    ];
    
    return $type_map[$ext] ?? 'text/plain';
}
?>