<?php
/**
 * AI Generator Class
 * Handles AI API integration and code generation
 */

class AIGenerator {
    private $openai_api_key;
    private $claude_api_key;
    private $default_provider;
    
    public function __construct() {
        $this->openai_api_key = OPENAI_API_KEY;
        $this->claude_api_key = CLAUDE_API_KEY;
        $this->default_provider = DEFAULT_AI_PROVIDER;
    }
    
    /**
     * Generate application based on user prompt
     */
    public function generate_app($prompt, $app_type = 'web', $framework = 'html') {
        try {
            // Analyze the prompt to determine the best approach
            $analysis = $this->analyze_prompt($prompt);
            
            // Generate application structure
            $structure = $this->generate_structure($prompt, $app_type, $framework, $analysis);
            
            // Generate code files
            $files = $this->generate_files($prompt, $structure, $app_type, $framework);
            
            return [
                'success' => true,
                'structure' => $structure,
                'files' => $files,
                'analysis' => $analysis
            ];
            
        } catch (Exception $e) {
            log_error("AI Generation failed: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Analyze user prompt to extract requirements
     */
    private function analyze_prompt($prompt) {
        $analysis = [
            'keywords' => [],
            'features' => [],
            'app_type' => 'web',
            'complexity' => 'medium',
            'components' => []
        ];
        
        // Convert to lowercase for analysis
        $prompt_lower = strtolower($prompt);
        
        // Extract keywords
        $common_words = ['app', 'application', 'web', 'website', 'create', 'build', 'make', 'design', 'the', 'a', 'an'];
        $words = explode(' ', preg_replace('/[^\w\s]/', '', $prompt_lower));
        $analysis['keywords'] = array_diff($words, $common_words);
        
        // Detect features
        $feature_patterns = [
            'login' => ['login', 'signin', 'authentication', 'auth', 'sign in'],
            'dashboard' => ['dashboard', 'admin', 'panel', 'control'],
            'database' => ['database', 'store', 'data', 'save', 'save data'],
            'search' => ['search', 'find', 'filter', 'lookup'],
            'ecommerce' => ['shop', 'store', 'buy', 'sell', 'cart', 'checkout', 'ecommerce'],
            'blog' => ['blog', 'posts', 'articles', 'news'],
            'portfolio' => ['portfolio', 'gallery', 'showcase', 'projects'],
            'contact' => ['contact', 'form', 'message', 'inquiry'],
            'social' => ['social', 'chat', 'comments', 'share', 'like'],
            'booking' => ['book', 'appointment', 'schedule', 'reservation'],
            'payment' => ['payment', 'billing', 'invoice', 'receipt', 'pay']
        ];
        
        foreach ($feature_patterns as $feature => $patterns) {
            foreach ($patterns as $pattern) {
                if (strpos($prompt_lower, $pattern) !== false) {
                    $analysis['features'][] = $feature;
                    break;
                }
            }
        }
        
        // Determine app type
        if (in_array('ecommerce', $analysis['features'])) {
            $analysis['app_type'] = 'ecommerce';
        } elseif (in_array('blog', $analysis['features'])) {
            $analysis['app_type'] = 'blog';
        } elseif (in_array('portfolio', $analysis['features'])) {
            $analysis['app_type'] = 'portfolio';
        } elseif (in_array('dashboard', $analysis['features'])) {
            $analysis['app_type'] = 'dashboard';
        } elseif (in_array('booking', $analysis['features'])) {
            $analysis['app_type'] = 'booking';
        } else {
            $analysis['app_type'] = 'web';
        }
        
        // Determine complexity
        $complexity_indicators = [
            'complex' => ['multiple', 'advanced', 'comprehensive', 'full-featured', 'enterprise'],
            'simple' => ['simple', 'basic', 'minimal', 'clean', 'minimalist']
        ];
        
        if ($this->contains_any($prompt_lower, $complexity_indicators['complex'])) {
            $analysis['complexity'] = 'complex';
        } elseif ($this->contains_any($prompt_lower, $complexity_indicators['simple'])) {
            $analysis['complexity'] = 'simple';
        }
        
        return $analysis;
    }
    
    /**
     * Check if string contains any of the given words
     */
    private function contains_any($string, $words) {
        foreach ($words as $word) {
            if (strpos($string, $word) !== false) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Generate application structure
     */
    private function generate_structure($prompt, $app_type, $framework, $analysis) {
        $base_structure = [
            'app_type' => $app_type,
            'framework' => $framework,
            'pages' => [],
            'components' => [],
            'features' => $analysis['features'],
            'complexity' => $analysis['complexity']
        ];
        
        // Basic pages based on app type
        switch ($app_type) {
            case 'ecommerce':
                $base_structure['pages'] = [
                    'index' => 'Homepage with featured products',
                    'products' => 'Product catalog',
                    'product-detail' => 'Individual product page',
                    'cart' => 'Shopping cart',
                    'checkout' => 'Checkout process',
                    'account' => 'User account dashboard'
                ];
                break;
                
            case 'blog':
                $base_structure['pages'] = [
                    'index' => 'Blog homepage',
                    'post' => 'Blog post detail',
                    'category' => 'Category archive',
                    'about' => 'About page',
                    'contact' => 'Contact form'
                ];
                break;
                
            case 'portfolio':
                $base_structure['pages'] = [
                    'index' => 'Portfolio homepage',
                    'projects' => 'Project gallery',
                    'project-detail' => 'Project detail',
                    'about' => 'About section',
                    'contact' => 'Contact form'
                ];
                break;
                
            case 'dashboard':
                $base_structure['pages'] = [
                    'dashboard' => 'Main dashboard',
                    'analytics' => 'Analytics page',
                    'users' => 'User management',
                    'settings' => 'Settings page',
                    'reports' => 'Reports section'
                ];
                break;
                
            default:
                $base_structure['pages'] = [
                    'index' => 'Homepage',
                    'about' => 'About page',
                    'contact' => 'Contact form',
                    'services' => 'Services page'
                ];
        }
        
        // Add common features as pages
        if (in_array('login', $analysis['features'])) {
            $base_structure['pages']['login'] = 'Login page';
            $base_structure['pages']['register'] = 'Registration page';
        }
        
        if (in_array('search', $analysis['features'])) {
            $base_structure['pages']['search'] = 'Search results page';
        }
        
        return $base_structure;
    }
    
    /**
     * Generate actual files for the application
     */
    private function generate_files($prompt, $structure, $app_type, $framework) {
        $files = [];
        
        // Generate main HTML files
        foreach ($structure['pages'] as $page_name => $description) {
            $files["pages/{$page_name}.html"] = $this->generate_html_page($page_name, $description, $structure);
        }
        
        // Generate CSS
        $files["assets/css/style.css"] = $this->generate_css($structure);
        
        // Generate JavaScript
        $files["assets/js/app.js"] = $this->generate_javascript($structure);
        
        // Generate configuration files
        $files["config.json"] = json_encode([
            'app_type' => $app_type,
            'framework' => $framework,
            'features' => $structure['features'],
            'pages' => array_keys($structure['pages'])
        ], JSON_PRETTY_PRINT);
        
        // Generate README
        $files["README.md"] = $this->generate_readme($structure, $prompt);
        
        // Generate index file
        $files["index.html"] = $this->generate_index($structure);
        
        return $files;
    }
    
    /**
     * Generate HTML page
     */
    private function generate_html_page($page_name, $description, $structure) {
        $page_title = ucwords(str_replace('-', ' ', $page_name));
        $app_type = $structure['app_type'];
        
        $html = "<!DOCTYPE html>\n";
        $html .= "<html lang=\"en\">\n";
        $html .= "<head>\n";
        $html .= "    <meta charset=\"UTF-8\">\n";
        $html .= "    <meta name=\"viewport\" content=\"width=device-width, initial-scale=1.0\">\n";
        $html .= "    <title>{$page_title} - AI Generated App</title>\n";
        $html .= "    <link rel=\"stylesheet\" href=\"assets/css/style.css\">\n";
        $html .= "    <link href=\"https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css\" rel=\"stylesheet\">\n";
        $html .= "</head>\n";
        $html .= "<body>\n";
        $html .= "    <header class=\"navbar\">\n";
        $html .= "        <div class=\"container\">\n";
        $html .= "            <div class=\"nav-brand\">\n";
        $html .= "                <h2>AI Generated App</h2>\n";
        $html .= "            </div>\n";
        $html .= "            <nav class=\"nav-menu\">\n";
        
        // Generate navigation menu
        foreach ($structure['pages'] as $nav_page => $nav_desc) {
            $nav_title = ucwords(str_replace('-', ' ', $nav_page));
            $active_class = ($nav_page === $page_name) ? ' class="active"' : '';
            $html .= "                <a href=\"{$nav_page}.html\"{$active_class}>{$nav_title}</a>\n";
        }
        
        $html .= "            </nav>\n";
        $html .= "        </div>\n";
        $html .= "    </header>\n";
        
        // Generate page content based on page type
        $html .= "    <main class=\"main-content\">\n";
        $html .= "        <div class=\"container\">\n";
        $html .= "            <h1 class=\"page-title\">{$page_title}</h1>\n";
        $html .= "            <div class=\"page-content\">\n";
        $html .= $this->generate_page_content($page_name, $structure);
        $html .= "            </div>\n";
        $html .= "        </div>\n";
        $html .= "    </main>\n";
        
        $html .= "    <footer class=\"footer\">\n";
        $html .= "        <div class=\"container\">\n";
        $html .= "            <p>&copy; 2024 AI Generated App. Built with AI technology.</p>\n";
        $html .= "        </div>\n";
        $html .= "    </footer>\n";
        
        $html .= "    <script src=\"assets/js/app.js\"></script>\n";
        $html .= "</body>\n";
        $html .= "</html>";
        
        return $html;
    }
    
    /**
     * Generate page-specific content
     */
    private function generate_page_content($page_name, $structure) {
        $content = '';
        
        switch ($page_name) {
            case 'index':
                $content = $this->generate_homepage_content($structure);
                break;
            case 'login':
                $content = $this->generate_login_content();
                break;
            case 'products':
            case 'post':
            case 'projects':
                $content = $this->generate_catalog_content($structure);
                break;
            case 'cart':
                $content = $this->generate_cart_content();
                break;
            default:
                $content = $this->generate_default_content($page_name);
        }
        
        return $content;
    }
    
    /**
     * Generate homepage content
     */
    private function generate_homepage_content($structure) {
        $app_type = $structure['app_type'];
        
        switch ($app_type) {
            case 'ecommerce':
                return '
                    <section class="hero">
                        <h2>Welcome to Our Store</h2>
                        <p>Discover amazing products for every need</p>
                        <a href="products.html" class="btn btn-primary">Shop Now</a>
                    </section>
                    <section class="featured-products">
                        <h3>Featured Products</h3>
                        <div class="product-grid">
                            <div class="product-card">
                                <div class="product-image">Product Image</div>
                                <h4>Sample Product</h4>
                                <p class="price">$29.99</p>
                                <button class="btn btn-secondary">Add to Cart</button>
                            </div>
                        </div>
                    </section>
                ';
                
            case 'blog':
                return '
                    <section class="blog-posts">
                        <h2>Latest Posts</h2>
                        <div class="post-list">
                            <article class="post-card">
                                <h3>Sample Blog Post</h3>
                                <p>Blog post content goes here...</p>
                                <a href="post.html" class="read-more">Read More</a>
                            </article>
                        </div>
                    </section>
                ';
                
            case 'portfolio':
                return '
                    <section class="hero">
                        <h2>My Creative Portfolio</h2>
                        <p>Showcasing my best work and projects</p>
                        <a href="projects.html" class="btn btn-primary">View Projects</a>
                    </section>
                    <section class="featured-work">
                        <h3>Featured Work</h3>
                        <div class="portfolio-grid">
                            <div class="portfolio-item">
                                <div class="project-image">Project Preview</div>
                                <h4>Sample Project</h4>
                                <p>Project description...</p>
                            </div>
                        </div>
                    </section>
                ';
                
            default:
                return '
                    <section class="hero">
                        <h2>Welcome to Our Website</h2>
                        <p>Your one-stop solution for everything you need</p>
                        <a href="about.html" class="btn btn-primary">Learn More</a>
                    </section>
                ';
        }
    }
    
    /**
     * Generate login content
     */
    private function generate_login_content() {
        return '
            <div class="auth-container">
                <div class="auth-card">
                    <h2>Login</h2>
                    <form class="auth-form">
                        <div class="form-group">
                            <label for="email">Email</label>
                            <input type="email" id="email" name="email" required>
                        </div>
                        <div class="form-group">
                            <label for="password">Password</label>
                            <input type="password" id="password" name="password" required>
                        </div>
                        <button type="submit" class="btn btn-primary">Login</button>
                        <p class="auth-link">Don\'t have an account? <a href="register.html">Register</a></p>
                    </form>
                </div>
            </div>
        ';
    }
    
    /**
     * Generate catalog content (products/posts/projects)
     */
    private function generate_catalog_content($structure) {
        $app_type = $structure['app_type'];
        $title = ucfirst($app_type);
        $button_text = in_array('ecommerce', $structure['features']) ? 'Add to Cart' : 'View Details';
        $item_class = in_array('ecommerce', $structure['features']) ? 'product-card' : 'post-card';
        
        return '
            <section class="catalog">
                <h2>Our ' . $title . '</h2>
                <div class="catalog-grid">
                    <div class="' . $item_class . '">
                        <div class="item-image">Image Placeholder</div>
                        <h4>Sample ' . rtrim($title, 's') . '</h4>
                        <p>Description of the ' . rtrim($title, 's') . '...</p>
                        <p class="price">$19.99</p>
                        <button class="btn btn-secondary">' . $button_text . '</button>
                    </div>
                </div>
            </section>
        ';
    }
    
    /**
     * Generate cart content
     */
    private function generate_cart_content() {
        return '
            <div class="cart-container">
                <h2>Shopping Cart</h2>
                <div class="cart-items">
                    <div class="cart-item">
                        <div class="item-info">
                            <h4>Sample Product</h4>
                            <p class="price">$19.99</p>
                            <p class="quantity">Quantity: 1</p>
                        </div>
                        <div class="item-actions">
                            <button class="btn btn-secondary">Update</button>
                            <button class="btn btn-danger">Remove</button>
                        </div>
                    </div>
                </div>
                <div class="cart-total">
                    <h3>Total: $19.99</h3>
                    <a href="checkout.html" class="btn btn-primary">Proceed to Checkout</a>
                </div>
            </div>
        ';
    }
    
    /**
     * Generate default page content
     */
    private function generate_default_content($page_name) {
        return '
            <section class="content-section">
                <h2>' . ucwords(str_replace('-', ' ', $page_name)) . '</h2>
                <p>This is the ' . $page_name . ' page. Content will be generated based on your requirements.</p>
            </section>
        ';
    }
    
    /**
     * Generate CSS
     */
    private function generate_css($structure) {
        return '/* AI Generated Styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: \'Inter\', -apple-system, BlinkMacSystemFont, sans-serif;
    line-height: 1.6;
    color: #333;
}

.container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 20px;
}

/* Navigation */
.navbar {
    background: #fff;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    padding: 1rem 0;
}

.navbar .container {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.nav-brand h2 {
    color: #3B82F6;
    font-size: 1.5rem;
}

.nav-menu {
    display: flex;
    gap: 2rem;
}

.nav-menu a {
    text-decoration: none;
    color: #333;
    font-weight: 500;
    transition: color 0.3s;
}

.nav-menu a:hover,
.nav-menu a.active {
    color: #3B82F6;
}

/* Main Content */
.main-content {
    min-height: calc(100vh - 140px);
    padding: 2rem 0;
}

.page-title {
    font-size: 2.5rem;
    margin-bottom: 2rem;
    color: #111827;
}

.page-content {
    max-width: 800px;
}

/* Hero Section */
.hero {
    text-align: center;
    padding: 4rem 0;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 12px;
    margin-bottom: 3rem;
}

.hero h2 {
    font-size: 3rem;
    margin-bottom: 1rem;
}

.hero p {
    font-size: 1.25rem;
    margin-bottom: 2rem;
    opacity: 0.9;
}

/* Buttons */
.btn {
    display: inline-block;
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
    font-size: 1rem;
}

.btn-primary {
    background: #3B82F6;
    color: white;
}

.btn-primary:hover {
    background: #1D4ED8;
    transform: translateY(-2px);
}

.btn-secondary {
    background: #6B7280;
    color: white;
}

.btn-secondary:hover {
    background: #4B5563;
}

.btn-danger {
    background: #EF4444;
    color: white;
}

.btn-danger:hover {
    background: #DC2626;
}

/* Cards */
.product-card,
.post-card,
.portfolio-item {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
    transition: transform 0.3s;
}

.product-card:hover,
.post-card:hover,
.portfolio-item:hover {
    transform: translateY(-4px);
}

.product-image,
.item-image,
.project-image,
.post-image {
    width: 100%;
    height: 200px;
    background: #F3F4F6;
    border-radius: 8px;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #9CA3AF;
    font-size: 0.9rem;
}

.price {
    font-size: 1.25rem;
    font-weight: 600;
    color: #3B82F6;
    margin: 0.5rem 0;
}

/* Grid Layouts */
.product-grid,
.portfolio-grid,
.catalog-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 2rem;
}

/* Forms */
.auth-form {
    max-width: 400px;
    margin: 0 auto;
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-group label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 500;
    color: #374151;
}

.form-group input {
    width: 100%;
    padding: 0.75rem;
    border: 2px solid #D1D5DB;
    border-radius: 8px;
    font-size: 1rem;
    transition: border-color 0.3s;
}

.form-group input:focus {
    outline: none;
    border-color: #3B82F6;
}

/* Footer */
.footer {
    background: #111827;
    color: white;
    text-align: center;
    padding: 2rem 0;
    margin-top: 4rem;
}

/* Responsive */
@media (max-width: 768px) {
    .navbar .container {
        flex-direction: column;
        gap: 1rem;
    }
    
    .nav-menu {
        gap: 1rem;
    }
    
    .hero h2 {
        font-size: 2rem;
    }
    
    .product-grid,
    .portfolio-grid,
    .catalog-grid {
        grid-template-columns: 1fr;
    }
}
';
    }
    
    /**
     * Generate JavaScript
     */
    private function generate_javascript($structure) {
        $app_type = $structure['app_type'];
        
        $js = '// AI Generated JavaScript
document.addEventListener(\'DOMContentLoaded\', function() {
    console.log(\'AI Generated App Loaded\');
    
    // Initialize based on app type
    ';
        
        if (in_array('ecommerce', $structure['features'])) {
            $js .= '
    initializeEcommerce();
    ';
        }
        
        if (in_array('login', $structure['features'])) {
            $js .= '
    initializeAuth();
    ';
        }
        
        $js .= '
    initializeNavigation();
    initializeAnimations();
});

function initializeNavigation() {
    // Add active class to current page
    const currentPage = window.location.pathname.split(\'/\').pop().replace(\'.html\', \'\') || \'index\';
    const navLinks = document.querySelectorAll(\'.nav-menu a\');
    
    navLinks.forEach(link => {
        const href = link.getAttribute(\'href\').replace(\'.html\', \'\');
        if (href === currentPage) {
            link.classList.add(\'active\');
        }
    });
}

function initializeAnimations() {
    // Simple fade-in animation for cards
    const cards = document.querySelectorAll(\'.product-card, .post-card, .portfolio-item\');
    
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = \'1\';
                entry.target.style.transform = \'translateY(0)\';
            }
        });
    });
    
    cards.forEach(card => {
        card.style.opacity = \'0\';
        card.style.transform = \'translateY(20px)\';
        card.style.transition = \'opacity 0.6s ease, transform 0.6s ease\';
        observer.observe(card);
    });
}
';
        
        // Add app-specific functions
        if (in_array('ecommerce', $structure['features'])) {
            $js .= '
function initializeEcommerce() {
    // Product card interactions
    const addToCartButtons = document.querySelectorAll(\'[data-action="add-to-cart"]\');
    
    addToCartButtons.forEach(button => {
        button.addEventListener(\'click\', function(e) {
            e.preventDefault();
            const productName = this.closest(\'.product-card\').querySelector(\'h4\').textContent;
            addToCart(productName);
        });
    });
}

function addToCart(productName) {
    // Simple cart functionality
    let cart = JSON.parse(localStorage.getItem(\'cart\')) || [];
    cart.push({
        name: productName,
        quantity: 1,
        addedAt: new Date().toISOString()
    });
    localStorage.setItem(\'cart\', JSON.stringify(cart));
    
    // Show success message
    showNotification(\'Product added to cart!\', \'success\');
}

function showNotification(message, type = \'info\') {
    const notification = document.createElement(\'div\');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.remove();
    }, 3000);
}
';
        }
        
        if (in_array('login', $structure['features'])) {
            $js .= '
function initializeAuth() {
    const loginForm = document.querySelector(\'.auth-form\');
    
    if (loginForm) {
        loginForm.addEventListener(\'submit\', function(e) {
            e.preventDefault();
            
            const email = this.querySelector(\'[name="email"]\').value;
            const password = this.querySelector(\'[name="password"]\').value;
            
            // Simple validation
            if (!email || !password) {
                showNotification(\'Please fill in all fields\', \'error\');
                return;
            }
            
            // Simulate login
            showNotification(\'Login functionality would be implemented here\', \'info\');
        });
    }
}
';
        }
        
        return $js;
    }
    
    /**
     * Generate README file
     */
    private function generate_readme($structure, $prompt) {
        $app_type = ucfirst($structure['app_type']);
        $features = implode(', ', $structure['features']);
        
        return "# AI Generated Application

## Description
This application was generated based on the following prompt:
> {$prompt}

## Application Type
**{$app_type}** application with the following features:
- " . implode("\n- ", array_map('ucfirst', $structure['features'])) . "

## Features
- " . implode("\n- ", $structure['features']) . "
- Modern, responsive design
- Cross-browser compatibility
- SEO optimized

## Pages
- " . implode("\n- ", array_map(function($key, $value) {
            return ucwords(str_replace('-', ' ', $key)) . ": " . $value;
        }, array_keys($structure['pages']), $structure['pages'])) . "

## Technologies Used
- HTML5
- CSS3 (Modern with Flexbox/Grid)
- Vanilla JavaScript
- Font Awesome Icons

## Setup Instructions
1. Upload all files to your web server
2. Ensure all paths are correct
3. Open `index.html` in your browser
4. Customize the content as needed

## Customization
- Edit HTML files in the `/pages` directory
- Modify styles in `/assets/css/style.css`
- Update functionality in `/assets/js/app.js`
- Replace placeholder content with your own

## Generated on
" . date('Y-m-d H:i:s') . "

## Notes
This application was generated using AI technology. While functional, you may want to enhance the features and styling to better suit your specific needs.
";
    }
    
    /**
     * Generate main index file
     */
    private function generate_index($structure) {
        $app_type = $structure['app_type'];
        
        switch ($app_type) {
            case 'ecommerce':
                $title = 'Store - AI Generated';
                break;
            case 'blog':
                $title = 'Blog - AI Generated';
                break;
            case 'portfolio':
                $title = 'Portfolio - AI Generated';
                break;
            case 'dashboard':
                $title = 'Dashboard - AI Generated';
                break;
            default:
                $title = 'AI Generated Website';
        }
        
        // For index, redirect to the main page of the app type
        switch ($app_type) {
            case 'ecommerce':
            case 'blog':
            case 'portfolio':
            case 'dashboard':
                return '<!DOCTYPE html>
<html>
<head>
    <title>' . $title . '</title>
    <meta http-equiv="refresh" content="0; url=pages/index.html">
    <script>window.location.href = "pages/index.html";</script>
</head>
<body>
    <p>Redirecting to your ' . $app_type . ' application...</p>
    <a href="pages/index.html">Click here if you are not redirected</a>
</body>
</html>';
                
            default:
                return '<!DOCTYPE html>
<html>
<head>
    <title>' . $title . '</title>
    <meta http-equiv="refresh" content="0; url=pages/index.html">
    <script>window.location.href = "pages/index.html";</script>
</head>
<body>
    <p>Redirecting to your application...</p>
    <a href="pages/index.html">Click here if you are not redirected</a>
</body>
</html>';
        }
    }
}
?>