<?php
/**
 * Core Functions
 * Contains all utility functions for the application
 */

/**
 * Check if user is logged in
 */
function is_logged_in() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Get current user ID
 */
function get_current_user_id() {
    return $_SESSION['user_id'] ?? null;
}

/**
 * Check if installation is complete
 */
function is_installation_complete() {
    $config_file = __DIR__ . '/../config/.installed';
    return file_exists($config_file);
}

/**
 * Mark installation as complete
 */
function mark_installation_complete() {
    $config_file = __DIR__ . '/../config/.installed';
    file_put_contents($config_file, date('Y-m-d H:i:s'));
}

/**
 * Sanitize input data
 */
function sanitize_input($data) {
    if (is_array($data)) {
        return array_map('sanitize_input', $data);
    }
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

/**
 * Validate email address
 */
function validate_email($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Generate secure password hash
 */
function hash_password($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Verify password
 */
function verify_password($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Generate CSRF token
 */
function generate_csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Verify CSRF token
 */
function verify_csrf_token($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Generate unique project ID
 */
function generate_project_id() {
    return 'proj_' . uniqid() . '_' . substr(md5(uniqid()), 0, 8);
}

/**
 * Format file size
 */
function format_file_size($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    return round($bytes, 2) . ' ' . $units[$pow];
}

/**
 * Get project status
 */
function get_project_status($project_id) {
    global $db;
    
    try {
        $stmt = $db->query("SELECT status FROM projects WHERE id = ?", [$project_id]);
        $result = $stmt->fetch();
        return $result['status'] ?? 'draft';
    } catch (Exception $e) {
        return 'error';
    }
}

/**
 * Update project status
 */
function update_project_status($project_id, $status) {
    global $db;
    
    try {
        $stmt = $db->query(
            "UPDATE projects SET status = ?, updated_at = NOW() WHERE id = ?",
            [$status, $project_id]
        );
        return $stmt->rowCount() > 0;
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Create directory recursively
 */
function create_directory($path) {
    if (!file_exists($path)) {
        return mkdir($path, 0755, true);
    }
    return true;
}

/**
 * Remove directory and all contents
 */
function remove_directory($path) {
    if (!file_exists($path)) {
        return false;
    }
    
    if (is_file($path) || is_link($path)) {
        return unlink($path);
    }
    
    $files = array_diff(scandir($path), ['.', '..']);
    foreach ($files as $file) {
        remove_directory($path . DIRECTORY_SEPARATOR . $file);
    }
    
    return rmdir($path);
}

/**
 * Compress files to ZIP
 */
function create_zip($files, $zip_path) {
    $zip = new ZipArchive();
    
    if ($zip->open($zip_path, ZipArchive::CREATE) !== TRUE) {
        return false;
    }
    
    foreach ($files as $file) {
        if (file_exists($file)) {
            $zip->addFile($file, basename($file));
        }
    }
    
    return $zip->close();
}

/**
 * Send email using PHP mail function
 */
function send_email($to, $subject, $message, $is_html = false) {
    $headers = [
        'From: ' . FROM_NAME . ' <' . FROM_EMAIL . '>',
        'Reply-To: ' . FROM_EMAIL,
        'X-Mailer: PHP/' . phpversion()
    ];
    
    if ($is_html) {
        $headers[] = 'Content-Type: text/html; charset=UTF-8';
    } else {
        $headers[] = 'Content-Type: text/plain; charset=UTF-8';
    }
    
    return mail($to, $subject, $message, implode("\r\n", $headers));
}

/**
 * Log error for debugging
 */
function log_error($message, $context = []) {
    if (!DEBUG_MODE) {
        return;
    }
    
    $log_message = date('Y-m-d H:i:s') . " - " . $message;
    if (!empty($context)) {
        $log_message .= " - Context: " . json_encode($context);
    }
    
    error_log($log_message . PHP_EOL, 3, APP_DIR . '/logs/error.log');
}

/**
 * Get client IP address
 */
function get_client_ip() {
    $ip_keys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
    foreach ($ip_keys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP, 
                    FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    return $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
}

/**
 * Rate limiting helper
 */
function check_rate_limit($ip, $action, $limit = 10, $window = 3600) {
    $cache_key = "rate_limit_{$ip}_{$action}";
    // Simple rate limiting using file cache
    $cache_file = __DIR__ . '/../cache/' . md5($cache_key) . '.json';
    
    $now = time();
    $data = [];
    
    if (file_exists($cache_file)) {
        $data = json_decode(file_get_contents($cache_file), true) ?? [];
    }
    
    // Clean old entries
    $data = array_filter($data, function($timestamp) use ($now, $window) {
        return ($now - $timestamp) < $window;
    });
    
    $current_requests = count($data);
    if ($current_requests >= $limit) {
        return false;
    }
    
    // Add current request
    $data[] = $now;
    file_put_contents($cache_file, json_encode($data));
    
    return true;
}

/**
 * Check if string is valid JSON
 */
function is_valid_json($string) {
    json_decode($string);
    return (json_last_error() == JSON_ERROR_NONE);
}

/**
 * Convert array to CSV
 */
function array_to_csv($data, $filename) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = fopen('php://output', 'w');
    
    // Add BOM for UTF-8
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    foreach ($data as $row) {
        fputcsv($output, $row);
    }
    
    fclose($output);
    exit;
}

/**
 * Get user by ID
 */
function get_user_by_id($user_id) {
    global $db;
    
    try {
        $stmt = $db->query("SELECT * FROM users WHERE id = ?", [$user_id]);
        return $stmt->fetch();
    } catch (Exception $e) {
        log_error("Failed to get user: " . $e->getMessage());
        return null;
    }
}

/**
 * Get user initials
 */
function get_user_initials($user) {
    if (!$user) return 'U';
    
    $name = $user['name'] ?? '';
    $email = $user['email'] ?? '';
    
    if ($name) {
        $words = explode(' ', $name);
        $initials = '';
        foreach ($words as $word) {
            $initials .= strtoupper(substr($word, 0, 1));
        }
        return substr($initials, 0, 2);
    }
    
    return strtoupper(substr($email, 0, 2));
}

/**
 * Get recent projects
 */
function get_recent_projects($user_id, $limit = 10) {
    global $db;
    
    try {
        $stmt = $db->query(
            "SELECT * FROM projects WHERE user_id = ? ORDER BY updated_at DESC LIMIT ?",
            [$user_id, $limit]
        );
        return $stmt->fetchAll();
    } catch (Exception $e) {
        log_error("Failed to get recent projects: " . $e->getMessage());
        return [];
    }
}

/**
 * Get user statistics
 */
function get_user_statistics($user_id) {
    global $db;
    
    try {
        $stats = [];
        
        // Total projects
        $stmt = $db->query("SELECT COUNT(*) as count FROM projects WHERE user_id = ?", [$user_id]);
        $stats['total_projects'] = $stmt->fetch()['count'];
        
        // Completed projects
        $stmt = $db->query("SELECT COUNT(*) as count FROM projects WHERE user_id = ? AND status = 'complete'", [$user_id]);
        $stats['completed_projects'] = $stmt->fetch()['count'];
        
        // Draft projects
        $stmt = $db->query("SELECT COUNT(*) as count FROM projects WHERE user_id = ? AND status = 'draft'", [$user_id]);
        $stats['draft_projects'] = $stmt->fetch()['count'];
        
        // Total downloads (this would be tracked separately)
        $stats['total_downloads'] = rand(0, 50); // Placeholder
        
        return $stats;
    } catch (Exception $e) {
        log_error("Failed to get user statistics: " . $e->getMessage());
        return [
            'total_projects' => 0,
            'completed_projects' => 0,
            'draft_projects' => 0,
            'total_downloads' => 0
        ];
    }
}

/**
 * Time ago helper
 */
function time_ago($datetime) {
    $time = time() - strtotime($datetime);
    
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . ' minutes ago';
    if ($time < 86400) return floor($time/3600) . ' hours ago';
    if ($time < 2592000) return floor($time/86400) . ' days ago';
    if ($time < 31536000) return floor($time/2592000) . ' months ago';
    return floor($time/31536000) . ' years ago';
}

/**
 * Create project
 */
function create_project($user_id, $title, $prompt, $app_type = 'web') {
    global $db;
    
    $project_id = generate_project_id();
    
    try {
        $stmt = $db->query(
            "INSERT INTO projects (id, user_id, title, prompt, app_type, status, created_at) VALUES (?, ?, ?, ?, ?, 'draft', NOW())",
            [$project_id, $user_id, $title, $prompt, $app_type]
        );
        
        return $project_id;
    } catch (Exception $e) {
        log_error("Failed to create project: " . $e->getMessage());
        return false;
    }
}
?>