<?php
/**
 * Export Page
 * Download generated applications
 */

include 'includes/header.php';

$project_id = $_GET['project'] ?? '';
$user_id = get_current_user_id();

if (empty($project_id)) {
    header('Location: ?page=projects');
    exit;
}

// Get project details
$project = get_project_by_id($project_id, $user_id);
if (!$project) {
    show_notification('Project not found', 'error');
    header('Location: ?page=projects');
    exit;
}

$files = get_project_files($project_id);
?>
<div class="export">
    <div class="export-header">
        <div class="container">
            <h1><i class="fas fa-download"></i> Export Project</h1>
            <p>Download your generated application for deployment</p>
        </div>
    </div>

    <div class="container">
        <div class="export-content">
            <!-- Project Info -->
            <div class="project-summary">
                <h2><?php echo htmlspecialchars($project['title']); ?></h2>
                <p class="project-type"><?php echo ucfirst($project['app_type']); ?> Application</p>
                <p class="project-description"><?php echo htmlspecialchars($project['description']); ?></p>
                
                <div class="project-stats">
                    <div class="stat">
                        <span class="stat-label">Status:</span>
                        <span class="stat-value status-<?php echo $project['status']; ?>"><?php echo ucfirst($project['status']); ?></span>
                    </div>
                    <div class="stat">
                        <span class="stat-label">Files:</span>
                        <span class="stat-value"><?php echo count($files); ?></span>
                    </div>
                    <div class="stat">
                        <span class="stat-label">Size:</span>
                        <span class="stat-value"><?php echo format_file_size($project['file_size']); ?></span>
                    </div>
                    <div class="stat">
                        <span class="stat-label">Created:</span>
                        <span class="stat-value"><?php echo date('M j, Y', strtotime($project['created_at'])); ?></span>
                    </div>
                </div>
            </div>

            <!-- Export Options -->
            <div class="export-options">
                <h3>Export Options</h3>
                
                <div class="export-form">
                    <div class="form-group">
                        <label for="export_format">Format</label>
                        <select id="export_format">
                            <option value="zip">ZIP Archive (Recommended)</option>
                            <option value="tar">TAR Archive</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="include_readme">
                            <input type="checkbox" id="include_readme" checked>
                            Include README.md with instructions
                        </label>
                    </div>
                    
                    <div class="form-group">
                        <label for="include_config">
                            <input type="checkbox" id="include_config" checked>
                            Include configuration file (config.json)
                        </label>
                    </div>
                    
                    <div class="form-group">
                        <label for="minify_code">
                            <input type="checkbox" id="minify_code">
                            Minify CSS and JavaScript files
                        </label>
                    </div>
                </div>
                
                <div class="export-actions">
                    <button id="exportBtn" class="btn btn-primary" onclick="startExport()">
                        <i class="fas fa-download"></i> Export Application
                    </button>
                    <a href="?page=projects" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> Back to Projects
                    </a>
                </div>
            </div>

            <!-- File List -->
            <div class="file-list">
                <h3>Generated Files (<?php echo count($files); ?> files)</h3>
                <div class="files">
                    <?php foreach ($files as $file): ?>
                        <div class="file-item">
                            <div class="file-info">
                                <i class="fas fa-file"></i>
                                <span class="file-path"><?php echo htmlspecialchars($file['file_path']); ?></span>
                            </div>
                            <div class="file-meta">
                                <span class="file-size"><?php echo format_file_size($file['file_size']); ?></span>
                                <span class="file-type"><?php echo strtoupper($file['file_type']); ?></span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Export Progress Modal -->
<div class="modal" id="exportModal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3><i class="fas fa-cog"></i> Preparing Export</h3>
        </div>
        <div class="modal-body">
            <div class="export-progress">
                <div class="progress-bar">
                    <div class="progress-fill" id="exportProgress"></div>
                </div>
                <div class="progress-status" id="exportStatus">Creating archive...</div>
            </div>
        </div>
    </div>
</div>

<style>
.export {
    min-height: 100vh;
    background: #F8FAFC;
}

.export-header {
    background: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
    color: white;
    padding: 4rem 0 3rem;
    text-align: center;
}

.export-header h1 {
    font-size: 3rem;
    margin-bottom: 1rem;
    font-weight: 700;
}

.export-header h1 i {
    margin-right: 1rem;
}

.export-header p {
    font-size: 1.25rem;
    opacity: 0.9;
}

.container {
    max-width: 1000px;
    margin: 0 auto;
    padding: 0 20px;
}

.export-content {
    background: white;
    border-radius: 16px;
    box-shadow: 0 10px 25px rgba(0,0,0,0.1);
    margin: -3rem 0 3rem;
    overflow: hidden;
}

.project-summary {
    padding: 2rem;
    border-bottom: 1px solid #E5E7EB;
}

.project-summary h2 {
    font-size: 2rem;
    color: #111827;
    margin-bottom: 0.5rem;
}

.project-type {
    color: #3B82F6;
    font-weight: 600;
    font-size: 1.125rem;
    margin-bottom: 1rem;
}

.project-description {
    color: #6B7280;
    line-height: 1.6;
    margin-bottom: 2rem;
}

.project-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
}

.stat {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.stat-label {
    color: #6B7280;
    font-size: 0.875rem;
    font-weight: 500;
}

.stat-value {
    color: #111827;
    font-weight: 600;
    font-size: 1.125rem;
}

.status-complete { color: #10B981; }
.status-draft { color: #F59E0B; }
.status-generating { color: #3B82F6; }
.status-error { color: #EF4444; }

.export-options {
    padding: 2rem;
    border-bottom: 1px solid #E5E7EB;
}

.export-options h3 {
    font-size: 1.5rem;
    color: #111827;
    margin-bottom: 1.5rem;
}

.export-form .form-group {
    margin-bottom: 1.5rem;
}

.export-form label {
    display: block;
    font-weight: 500;
    color: #374151;
    margin-bottom: 0.5rem;
}

.export-form input[type="checkbox"] {
    margin-right: 0.5rem;
}

.export-form select {
    width: 100%;
    padding: 0.75rem;
    border: 2px solid #D1D5DB;
    border-radius: 8px;
    font-size: 1rem;
}

.export-actions {
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
    margin-top: 2rem;
}

.file-list {
    padding: 2rem;
}

.file-list h3 {
    font-size: 1.5rem;
    color: #111827;
    margin-bottom: 1.5rem;
}

.files {
    max-height: 400px;
    overflow-y: auto;
    border: 1px solid #E5E7EB;
    border-radius: 8px;
    background: #F9FAFB;
}

.file-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1rem;
    border-bottom: 1px solid #E5E7EB;
}

.file-item:last-child {
    border-bottom: none;
}

.file-info {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    flex: 1;
}

.file-info i {
    color: #6B7280;
}

.file-path {
    font-family: 'Courier New', monospace;
    font-size: 0.875rem;
    color: #374151;
}

.file-meta {
    display: flex;
    gap: 1rem;
    font-size: 0.75rem;
    color: #6B7280;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
    text-decoration: none;
    font-size: 1rem;
}

.btn-primary {
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    color: white;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #1D4ED8, #1E40AF);
    transform: translateY(-2px);
}

.btn-secondary {
    background: #6B7280;
    color: white;
}

.btn-secondary:hover {
    background: #4B5563;
}

/* Modal */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
}

.modal-content {
    background: white;
    border-radius: 12px;
    max-width: 400px;
    width: 90%;
    overflow: hidden;
    box-shadow: 0 25px 50px rgba(0,0,0,0.3);
}

.modal-header {
    padding: 1.5rem;
    border-bottom: 1px solid #E5E7EB;
    text-align: center;
}

.modal-body {
    padding: 2rem;
    text-align: center;
}

.export-progress {
    width: 100%;
}

.progress-bar {
    width: 100%;
    height: 8px;
    background: #E5E7EB;
    border-radius: 4px;
    margin-bottom: 1rem;
    overflow: hidden;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    border-radius: 4px;
    width: 0%;
    transition: width 0.5s ease;
}

.progress-status {
    color: #6B7280;
    font-size: 0.875rem;
}

/* Responsive */
@media (max-width: 768px) {
    .export-header h1 {
        font-size: 2rem;
    }
    
    .export-content {
        margin: -2rem 0 2rem;
    }
    
    .project-stats {
        grid-template-columns: 1fr;
    }
    
    .export-actions {
        flex-direction: column;
    }
    
    .file-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.5rem;
    }
}
</style>

<script>
function startExport() {
    const exportBtn = document.getElementById('exportBtn');
    const exportModal = document.getElementById('exportModal');
    
    // Show loading state
    exportBtn.disabled = true;
    exportBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Preparing...';
    exportModal.style.display = 'flex';
    
    // Start export process
    const exportData = {
        project_id: '<?php echo $project_id; ?>',
        format: document.getElementById('export_format').value,
        include_readme: document.getElementById('include_readme').checked,
        include_config: document.getElementById('include_config').checked,
        minify_code: document.getElementById('minify_code').checked,
        csrf_token: csrfToken
    };
    
    // Simulate progress
    let progress = 0;
    const progressInterval = setInterval(() => {
        progress += Math.random() * 20;
        if (progress > 90) progress = 90;
        
        document.getElementById('exportProgress').style.width = progress + '%';
        
        if (progress < 30) {
            document.getElementById('exportStatus').textContent = 'Collecting files...';
        } else if (progress < 60) {
            document.getElementById('exportStatus').textContent = 'Creating archive...';
        } else if (progress < 90) {
            document.getElementById('exportStatus').textContent = 'Compressing...';
        }
    }, 500);
    
    // Make API call
    fetch('api/index.php?action=export_project', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': csrfToken
        },
        body: JSON.stringify(exportData)
    })
    .then(response => response.json())
    .then(result => {
        clearInterval(progressInterval);
        
        if (result.success) {
            document.getElementById('exportProgress').style.width = '100%';
            document.getElementById('exportStatus').textContent = 'Download ready!';
            
            // Start download
            setTimeout(() => {
                window.location.href = result.download_url;
                
                setTimeout(() => {
                    exportModal.style.display = 'none';
                    exportBtn.disabled = false;
                    exportBtn.innerHTML = '<i class="fas fa-download"></i> Export Application';
                }, 2000);
            }, 1000);
        } else {
            throw new Error(result.error || 'Export failed');
        }
    })
    .catch(error => {
        clearInterval(progressInterval);
        exportModal.style.display = 'none';
        exportBtn.disabled = false;
        exportBtn.innerHTML = '<i class="fas fa-download"></i> Export Application';
        showNotification('Export failed: ' + error.message, 'error');
    });
}
</script>

<?php
function get_project_by_id($project_id, $user_id) {
    global $db;
    
    try {
        $stmt = $db->query("SELECT * FROM projects WHERE id = ? AND user_id = ?", [$project_id, $user_id]);
        return $stmt->fetch();
    } catch (Exception $e) {
        log_error("Failed to get project: " . $e->getMessage());
        return null;
    }
}

function get_project_files($project_id) {
    global $db;
    
    try {
        $stmt = $db->query("SELECT * FROM project_files WHERE project_id = ? ORDER BY file_path", [$project_id]);
        return $stmt->fetchAll();
    } catch (Exception $e) {
        log_error("Failed to get project files: " . $e->getMessage());
        return [];
    }
}
?>

<?php
include 'includes/footer.php';
?>