<?php
/**
 * AI Generator Page
 * Main interface for generating web applications
 */

include 'includes/header.php';

// Get user templates
$templates = get_app_templates();
?>
<div class="generator">
    <div class="generator-header">
        <div class="container">
            <h1><i class="fas fa-magic"></i> Create New Application</h1>
            <p>Describe your web application and let AI build it for you</p>
        </div>
    </div>

    <div class="container">
        <div class="generator-content">
            <!-- Templates Section -->
            <div class="templates-section">
                <h2>Quick Start Templates</h2>
                <p>Choose from pre-built templates to get started quickly</p>
                
                <div class="templates-grid">
                    <?php foreach ($templates as $template): ?>
                        <div class="template-card" onclick="useTemplate('<?php echo htmlspecialchars($template['prompt_template']); ?>', '<?php echo htmlspecialchars($template['name']); ?>')">
                            <div class="template-icon">
                                <i class="<?php echo htmlspecialchars($template['icon']); ?>"></i>
                            </div>
                            <h3><?php echo htmlspecialchars($template['name']); ?></h3>
                            <p><?php echo htmlspecialchars($template['description']); ?></p>
                            <div class="template-category"><?php echo htmlspecialchars($template['category']); ?></div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Generator Form -->
            <div class="generator-form-section">
                <h2>Describe Your Application</h2>
                
                <form id="generatorForm" class="generator-form">
                    <div class="form-group">
                        <label for="project_title">
                            <i class="fas fa-tag"></i> Project Title
                        </label>
                        <input type="text" id="project_title" name="title" 
                               placeholder="e.g., My Online Store, Personal Blog, Business Website"
                               required>
                    </div>

                    <div class="form-group">
                        <label for="app_type">
                            <i class="fas fa-layer-group"></i> Application Type
                        </label>
                        <select id="app_type" name="app_type" required>
                            <option value="web">General Website</option>
                            <option value="ecommerce">E-commerce Store</option>
                            <option value="blog">Blog/News Site</option>
                            <option value="portfolio">Portfolio Website</option>
                            <option value="dashboard">Business Dashboard</option>
                            <option value="landing">Landing Page</option>
                            <option value="business">Business Website</option>
                            <option value="restaurant">Restaurant Website</option>
                            <option value="booking">Booking/Reservation System</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="prompt">
                            <i class="fas fa-comment-dots"></i> Describe Your Application
                        </label>
                        <textarea id="prompt" name="prompt" rows="6" 
                                  placeholder="Describe what you want to build. Be specific about features, design style, and functionality. For example: 'Create a modern e-commerce store with product catalog, shopping cart, and checkout. Use a clean, professional design with blue and white colors. Include user registration, product search, and payment integration.'"
                                  required></textarea>
                        <div class="prompt-tips">
                            <h4>💡 Tips for better results:</h4>
                            <ul>
                                <li>Be specific about features you want (login, search, forms, etc.)</li>
                                <li>Mention your preferred color scheme or design style</li>
                                <li>Describe the target audience (business, personal, etc.)</li>
                                <li>Include any specific functionality you need</li>
                            </ul>
                        </div>
                    </div>

                    <div class="form-group">
                        <label for="features">
                            <i class="fas fa-list"></i> Additional Features (Optional)
                        </label>
                        <div class="features-grid">
                            <label class="feature-checkbox">
                                <input type="checkbox" name="features[]" value="login">
                                <span class="checkmark"></span>
                                User Login/Registration
                            </label>
                            <label class="feature-checkbox">
                                <input type="checkbox" name="features[]" value="search">
                                <span class="checkmark"></span>
                                Search Functionality
                            </label>
                            <label class="feature-checkbox">
                                <input type="checkbox" name="features[]" value="contact">
                                <span class="checkmark"></span>
                                Contact Form
                            </label>
                            <label class="feature-checkbox">
                                <input type="checkbox" name="features[]" value="blog">
                                <span class="checkmark"></span>
                                Blog/News Section
                            </label>
                            <label class="feature-checkbox">
                                <input type="checkbox" name="features[]" value="gallery">
                                <span class="checkmark"></span>
                                Image Gallery
                            </label>
                            <label class="feature-checkbox">
                                <input type="checkbox" name="features[]" value="map">
                                <span class="checkmark"></span>
                                Map Integration
                            </label>
                            <label class="feature-checkbox">
                                <input type="checkbox" name="features[]" value="social">
                                <span class="checkmark"></span>
                                Social Media Links
                            </label>
                            <label class="feature-checkbox">
                                <input type="checkbox" name="features[]" value="analytics">
                                <span class="checkmark"></span>
                                Analytics Dashboard
                            </label>
                        </div>
                    </div>

                    <div class="form-actions">
                        <button type="button" class="btn btn-secondary" onclick="previewPrompt()">
                            <i class="fas fa-eye"></i> Preview Description
                        </button>
                        <button type="submit" class="btn btn-primary" id="generateBtn">
                            <i class="fas fa-magic"></i> Generate Application
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Generation Modal -->
    <div class="modal" id="generationModal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h3><i class="fas fa-cogs"></i> Generating Your Application</h3>
            </div>
            <div class="modal-body">
                <div class="generation-progress">
                    <div class="progress-bar">
                        <div class="progress-fill" id="progressFill"></div>
                    </div>
                    <div class="generation-status" id="generationStatus">Analyzing your requirements...</div>
                    <div class="generation-steps" id="generationSteps">
                        <div class="step active">Analyzing Requirements</div>
                        <div class="step">Generating Structure</div>
                        <div class="step">Building Code</div>
                        <div class="step">Finalizing</div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.generator {
    min-height: 100vh;
    background: #F8FAFC;
}

.generator-header {
    background: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
    color: white;
    padding: 4rem 0 3rem;
    text-align: center;
}

.generator-header h1 {
    font-size: 3rem;
    margin-bottom: 1rem;
    font-weight: 700;
}

.generator-header h1 i {
    margin-right: 1rem;
}

.generator-header p {
    font-size: 1.25rem;
    opacity: 0.9;
}

.container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 20px;
}

.generator-content {
    display: grid;
    grid-template-columns: 1fr 2fr;
    gap: 3rem;
    margin: -3rem 0 3rem;
}

/* Templates Section */
.templates-section {
    background: white;
    border-radius: 16px;
    padding: 2rem;
    box-shadow: 0 10px 25px rgba(0,0,0,0.1);
    height: fit-content;
}

.templates-section h2 {
    font-size: 1.5rem;
    margin-bottom: 0.5rem;
    color: #111827;
}

.templates-section p {
    color: #6B7280;
    margin-bottom: 2rem;
}

.templates-grid {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.template-card {
    background: #F9FAFB;
    border: 2px solid transparent;
    border-radius: 12px;
    padding: 1.5rem;
    cursor: pointer;
    transition: all 0.3s;
}

.template-card:hover {
    border-color: #3B82F6;
    background: #EBF8FF;
    transform: translateY(-2px);
}

.template-icon {
    width: 40px;
    height: 40px;
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.25rem;
    margin-bottom: 1rem;
}

.template-card h3 {
    font-size: 1.125rem;
    color: #111827;
    margin-bottom: 0.5rem;
}

.template-card p {
    color: #6B7280;
    font-size: 0.875rem;
    line-height: 1.5;
    margin-bottom: 0.5rem;
}

.template-category {
    display: inline-block;
    background: #E5E7EB;
    color: #6B7280;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 500;
}

/* Generator Form */
.generator-form-section {
    background: white;
    border-radius: 16px;
    padding: 2rem;
    box-shadow: 0 10px 25px rgba(0,0,0,0.1);
}

.generator-form-section h2 {
    font-size: 1.5rem;
    margin-bottom: 0.5rem;
    color: #111827;
}

.form-group {
    margin-bottom: 2rem;
}

.form-group label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.75rem;
}

.form-group label i {
    color: #3B82F6;
}

.form-group input,
.form-group select,
.form-group textarea {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid #D1D5DB;
    border-radius: 8px;
    font-size: 1rem;
    transition: all 0.3s;
    background: #F9FAFB;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #3B82F6;
    background: white;
}

.prompt-tips {
    margin-top: 1rem;
    padding: 1rem;
    background: #F0F9FF;
    border: 1px solid #BAE6FD;
    border-radius: 8px;
    font-size: 0.875rem;
}

.prompt-tips h4 {
    color: #0C4A6E;
    margin-bottom: 0.5rem;
}

.prompt-tips ul {
    margin: 0;
    padding-left: 1.5rem;
    color: #0C4A6E;
}

.prompt-tips li {
    margin-bottom: 0.25rem;
}

/* Features Grid */
.features-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
}

.feature-checkbox {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.75rem;
    background: #F9FAFB;
    border: 1px solid #E5E7EB;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s;
}

.feature-checkbox:hover {
    background: #F3F4F6;
}

.feature-checkbox input[type="checkbox"] {
    width: auto;
    margin: 0;
}

.checkmark {
    width: 20px;
    height: 20px;
    border: 2px solid #D1D5DB;
    border-radius: 4px;
    position: relative;
    flex-shrink: 0;
}

.feature-checkbox input[type="checkbox"]:checked + .checkmark {
    background: #3B82F6;
    border-color: #3B82F6;
}

.feature-checkbox input[type="checkbox"]:checked + .checkmark::after {
    content: '✓';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    color: white;
    font-size: 12px;
    font-weight: bold;
}

/* Form Actions */
.form-actions {
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
    text-decoration: none;
    font-size: 1rem;
}

.btn-primary {
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    color: white;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #1D4ED8, #1E40AF);
    transform: translateY(-2px);
}

.btn-secondary {
    background: #6B7280;
    color: white;
}

.btn-secondary:hover {
    background: #4B5563;
}

/* Modal */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
}

.modal-content {
    background: white;
    border-radius: 16px;
    max-width: 500px;
    width: 90%;
    overflow: hidden;
    box-shadow: 0 25px 50px rgba(0,0,0,0.3);
}

.modal-header {
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    color: white;
    padding: 1.5rem;
    text-align: center;
}

.modal-body {
    padding: 2rem;
}

.generation-progress {
    text-align: center;
}

.progress-bar {
    width: 100%;
    height: 8px;
    background: #E5E7EB;
    border-radius: 4px;
    margin-bottom: 1.5rem;
    overflow: hidden;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    border-radius: 4px;
    width: 0%;
    transition: width 0.5s ease;
}

.generation-status {
    font-size: 1.125rem;
    color: #111827;
    margin-bottom: 1.5rem;
    font-weight: 500;
}

.generation-steps {
    display: flex;
    justify-content: space-between;
    margin-top: 2rem;
}

.step {
    flex: 1;
    text-align: center;
    padding: 0.5rem;
    font-size: 0.875rem;
    color: #9CA3AF;
    border-bottom: 2px solid #E5E7EB;
}

.step.active {
    color: #3B82F6;
    border-bottom-color: #3B82F6;
}

/* Responsive */
@media (max-width: 768px) {
    .generator-header h1 {
        font-size: 2rem;
    }
    
    .generator-content {
        grid-template-columns: 1fr;
        gap: 2rem;
        margin: -2rem 0 2rem;
    }
    
    .form-actions {
        flex-direction: column;
    }
    
    .features-grid {
        grid-template-columns: 1fr;
    }
    
    .generation-steps {
        flex-direction: column;
        gap: 0.5rem;
    }
}
</style>

<script>
function useTemplate(template, name) {
    document.getElementById('prompt').value = template;
    document.getElementById('project_title').value = name;
    showNotification('Template loaded! You can modify the description as needed.', 'info');
}

function previewPrompt() {
    const title = document.getElementById('project_title').value;
    const prompt = document.getElementById('prompt').value;
    const appType = document.getElementById('app_type').value;
    
    const preview = `Project: ${title}\nType: ${appType}\n\nDescription:\n${prompt}`;
    
    alert('Preview:\n\n' + preview);
}

document.getElementById('generatorForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const title = document.getElementById('project_title').value;
    const prompt = document.getElementById('prompt').value;
    const appType = document.getElementById('app_type').value;
    
    // Show generation modal
    document.getElementById('generationModal').style.display = 'flex';
    document.getElementById('generateBtn').disabled = true;
    
    // Start generation process
    startGeneration({ title, prompt, app_type: appType });
});

function startGeneration(data) {
    const steps = [
        { name: 'Analyzing Requirements', duration: 2000 },
        { name: 'Generating Structure', duration: 3000 },
        { name: 'Building Code', duration: 4000 },
        { name: 'Finalizing', duration: 2000 }
    ];
    
    let currentStep = 0;
    const statusEl = document.getElementById('generationStatus');
    const progressEl = document.getElementById('progressFill');
    const stepsEl = document.getElementById('generationSteps');
    
    function nextStep() {
        if (currentStep < steps.length) {
            const step = steps[currentStep];
            const progress = (currentStep / steps.length) * 100;
            
            statusEl.textContent = step.name;
            progressEl.style.width = progress + '%';
            
            // Update step indicators
            const stepElements = stepsEl.querySelectorAll('.step');
            stepElements.forEach((el, index) => {
                el.classList.toggle('active', index === currentStep);
            });
            
            currentStep++;
            
            setTimeout(nextStep, step.duration);
        } else {
            // Generation complete
            progressEl.style.width = '100%';
            statusEl.textContent = 'Generation Complete!';
            
            setTimeout(() => {
                submitGeneration(data);
            }, 1000);
        }
    }
    
    nextStep();
}

function submitGeneration(data) {
    // Submit form data via AJAX
    fetch('api/generate.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': csrfToken
        },
        body: JSON.stringify(data)
    })
    .then(response => response.json())
    .then(result => {
        if (result.success) {
            document.getElementById('generationModal').style.display = 'none';
            showNotification('Application generated successfully!', 'success');
            
            // Redirect to project
            setTimeout(() => {
                window.location.href = '?page=projects&action=edit&project=' + result.project_id;
            }, 2000);
        } else {
            throw new Error(result.error || 'Generation failed');
        }
    })
    .catch(error => {
        document.getElementById('generationModal').style.display = 'none';
        document.getElementById('generateBtn').disabled = false;
        showNotification('Generation failed: ' + error.message, 'error');
    });
}
</script>

<?php
function get_app_templates() {
    global $db;
    
    try {
        $stmt = $db->query("SELECT * FROM templates WHERE is_active = 1 ORDER BY sort_order ASC LIMIT 8");
        return $stmt->fetchAll();
    } catch (Exception $e) {
        log_error("Failed to get templates: " . $e->getMessage());
        return [];
    }
}
?>

<?php
include 'includes/footer.php';
?>