<?php
/**
 * Projects Management Page
 */

include 'includes/header.php';

$user_id = get_current_user_id();
$action = $_GET['action'] ?? '';
$project_id = $_GET['project'] ?? '';

// Handle actions
if ($action === 'delete' && $project_id && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verify_csrf_form()) {
        show_notification('Invalid security token', 'error');
    } else {
        $deleted = delete_project($project_id, $user_id);
        if ($deleted) {
            show_notification('Project deleted successfully', 'success');
        } else {
            show_notification('Failed to delete project', 'error');
        }
    }
}

// Get user projects
$projects = get_user_projects($user_id);
?>
<div class="projects">
    <div class="projects-header">
        <div class="container">
            <h1><i class="fas fa-folder"></i> My Projects</h1>
            <p>Manage and organize your generated applications</p>
        </div>
    </div>

    <div class="container">
        <!-- Quick Actions -->
        <div class="quick-actions-bar">
            <a href="?page=generator" class="btn btn-primary">
                <i class="fas fa-plus"></i> New Project
            </a>
            <a href="?page=templates" class="btn btn-secondary">
                <i class="fas fa-template"></i> Browse Templates
            </a>
        </div>

        <!-- Projects Grid -->
        <?php if (empty($projects)): ?>
            <div class="empty-state">
                <i class="fas fa-project-diagram"></i>
                <h3>No projects yet</h3>
                <p>Create your first AI-powered application to get started</p>
                <a href="?page=generator" class="btn btn-primary">Create First Project</a>
            </div>
        <?php else: ?>
            <div class="projects-grid">
                <?php foreach ($projects as $project): ?>
                    <div class="project-card">
                        <div class="project-header">
                            <h3><?php echo htmlspecialchars($project['title']); ?></h3>
                            <span class="status-badge status-<?php echo $project['status']; ?>">
                                <?php echo ucfirst($project['status']); ?>
                            </span>
                        </div>
                        
                        <div class="project-info">
                            <div class="project-type">
                                <i class="fas fa-tag"></i>
                                <?php echo ucfirst($project['app_type']); ?>
                            </div>
                            <div class="project-date">
                                <i class="fas fa-calendar"></i>
                                <?php echo time_ago($project['created_at']); ?>
                            </div>
                            <?php if ($project['file_size'] > 0): ?>
                            <div class="project-size">
                                <i class="fas fa-hdd"></i>
                                <?php echo format_file_size($project['file_size']); ?>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <p class="project-description">
                            <?php echo htmlspecialchars(substr($project['description'] ?? 'No description', 0, 120)); ?>
                            <?php echo strlen($project['description'] ?? '') > 120 ? '...' : ''; ?>
                        </p>
                        
                        <div class="project-actions">
                            <?php if ($project['status'] === 'complete'): ?>
                                <a href="?page=export&project=<?php echo $project['id']; ?>" class="btn btn-sm btn-primary">
                                    <i class="fas fa-download"></i> Export
                                </a>
                                <a href="#" class="btn btn-sm btn-secondary" onclick="previewProject('<?php echo $project['id']; ?>')">
                                    <i class="fas fa-eye"></i> Preview
                                </a>
                            <?php elseif ($project['status'] === 'generating'): ?>
                                <button class="btn btn-sm btn-secondary" disabled>
                                    <i class="fas fa-spinner fa-spin"></i> Generating...
                                </button>
                            <?php else: ?>
                                <a href="?page=generator&edit=<?php echo $project['id']; ?>" class="btn btn-sm btn-primary">
                                    <i class="fas fa-edit"></i> Edit
                                </a>
                            <?php endif; ?>
                            
                            <button class="btn btn-sm btn-danger" onclick="deleteProject('<?php echo $project['id']; ?>', '<?php echo htmlspecialchars($project['title']); ?>')">
                                <i class="fas fa-trash"></i> Delete
                            </button>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal" id="deleteModal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3><i class="fas fa-exclamation-triangle"></i> Delete Project</h3>
        </div>
        <div class="modal-body">
            <p>Are you sure you want to delete "<span id="deleteProjectTitle"></span>"?</p>
            <p class="text-danger">This action cannot be undone.</p>
        </div>
        <div class="modal-footer">
            <form method="POST" id="deleteForm">
                <?php echo csrf_field(); ?>
                <button type="button" class="btn btn-secondary" onclick="closeDeleteModal()">Cancel</button>
                <button type="submit" class="btn btn-danger">
                    <i class="fas fa-trash"></i> Delete
                </button>
            </form>
        </div>
    </div>
</div>

<style>
.projects {
    min-height: 100vh;
    background: #F8FAFC;
}

.projects-header {
    background: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
    color: white;
    padding: 4rem 0 3rem;
    text-align: center;
}

.projects-header h1 {
    font-size: 3rem;
    margin-bottom: 1rem;
    font-weight: 700;
}

.projects-header h1 i {
    margin-right: 1rem;
}

.projects-header p {
    font-size: 1.25rem;
    opacity: 0.9;
}

.container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 20px;
}

.quick-actions-bar {
    display: flex;
    gap: 1rem;
    margin: -3rem 0 3rem;
    justify-content: center;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
    text-decoration: none;
    font-size: 1rem;
}

.btn-primary {
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    color: white;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #1D4ED8, #1E40AF);
    transform: translateY(-2px);
    text-decoration: none;
    color: white;
}

.btn-secondary {
    background: #6B7280;
    color: white;
}

.btn-secondary:hover {
    background: #4B5563;
    text-decoration: none;
    color: white;
}

.btn-sm {
    padding: 0.5rem 1rem;
    font-size: 0.875rem;
}

.btn-danger {
    background: #EF4444;
    color: white;
}

.btn-danger:hover {
    background: #DC2626;
    color: white;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    background: white;
    border-radius: 16px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
}

.empty-state i {
    font-size: 4rem;
    color: #D1D5DB;
    margin-bottom: 1.5rem;
}

.empty-state h3 {
    font-size: 1.5rem;
    color: #6B7280;
    margin-bottom: 1rem;
}

.empty-state p {
    color: #9CA3AF;
    margin-bottom: 2rem;
}

.projects-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 2rem;
    margin: 2rem 0;
}

.project-card {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
    transition: transform 0.3s;
}

.project-card:hover {
    transform: translateY(-2px);
}

.project-header {
    display: flex;
    justify-content: space-between;
    align-items: start;
    margin-bottom: 1rem;
}

.project-header h3 {
    font-size: 1.25rem;
    color: #111827;
    margin: 0;
    flex: 1;
}

.status-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
}

.status-draft { background: #FEF3C7; color: #92400E; }
.status-generating { background: #DBEAFE; color: #1E40AF; }
.status-complete { background: #D1FAE5; color: #065F46; }
.status-error { background: #FEE2E2; color: #991B1B; }

.project-info {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
    margin-bottom: 1rem;
    font-size: 0.875rem;
    color: #6B7280;
}

.project-info > div {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.project-info i {
    color: #9CA3AF;
    width: 16px;
}

.project-description {
    color: #6B7280;
    margin-bottom: 1.5rem;
    line-height: 1.6;
    font-size: 0.875rem;
}

.project-actions {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.project-actions .btn {
    flex: 1;
    min-width: 80px;
    justify-content: center;
}

/* Modal Styles */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
}

.modal-content {
    background: white;
    border-radius: 12px;
    max-width: 500px;
    width: 90%;
    overflow: hidden;
    box-shadow: 0 25px 50px rgba(0,0,0,0.3);
}

.modal-header {
    padding: 1.5rem;
    border-bottom: 1px solid #E5E7EB;
}

.modal-header h3 {
    margin: 0;
    color: #EF4444;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.modal-body {
    padding: 1.5rem;
}

.modal-body p {
    margin-bottom: 1rem;
}

.text-danger {
    color: #EF4444;
    font-size: 0.875rem;
}

.modal-footer {
    padding: 1.5rem;
    border-top: 1px solid #E5E7EB;
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
}

.modal-footer form {
    display: flex;
    gap: 1rem;
}

/* Responsive */
@media (max-width: 768px) {
    .projects-header h1 {
        font-size: 2rem;
    }
    
    .quick-actions-bar {
        flex-direction: column;
        align-items: center;
    }
    
    .projects-grid {
        grid-template-columns: 1fr;
    }
    
    .project-header {
        flex-direction: column;
        gap: 1rem;
    }
    
    .project-actions {
        flex-direction: column;
    }
}
</style>

<script>
function deleteProject(projectId, projectTitle) {
    document.getElementById('deleteProjectTitle').textContent = projectTitle;
    document.getElementById('deleteForm').action = `?page=projects&action=delete&project=${projectId}`;
    document.getElementById('deleteModal').style.display = 'flex';
}

function closeDeleteModal() {
    document.getElementById('deleteModal').style.display = 'none';
}

function previewProject(projectId) {
    showNotification('Preview functionality coming soon!', 'info');
    // In a real implementation, this would show a preview modal
    // or redirect to a preview page
}
</script>

<?php
function get_user_projects($user_id) {
    global $db;
    
    try {
        $stmt = $db->query(
            "SELECT * FROM projects WHERE user_id = ? ORDER BY updated_at DESC",
            [$user_id]
        );
        return $stmt->fetchAll();
    } catch (Exception $e) {
        log_error("Failed to get user projects: " . $e->getMessage());
        return [];
    }
}
?>

<?php
include 'includes/footer.php';
?>