<?php
/**
 * JWT Authentication Middleware
 * Madrasa Management System
 */

class JWT {
    public static function encode($payload) {
        $header = json_encode(['typ' => 'JWT', 'alg' => Config::$jwt_algorithm]);
        $payload = json_encode($payload);
        
        $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
        $base64Payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
        
        $signature = hash_hmac('sha256', $base64Header . "." . $base64Payload, Config::$jwt_secret, true);
        $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        return $base64Header . "." . $base64Payload . "." . $base64Signature;
    }

    public static function decode($token) {
        $parts = explode('.', $token);
        if (count($parts) !== 3) {
            return false;
        }

        $header = $parts[0];
        $payload = $parts[1];
        $signature = $parts[2];

        $validSignature = str_replace(['+', '/', '='], ['-', '_', ''], 
            base64_encode(hash_hmac('sha256', $header . "." . $payload, Config::$jwt_secret, true))
        );

        if (!hash_equals($signature, $validSignature)) {
            return false;
        }

        $payload = json_decode(base64_decode(str_replace(['-', '_'], ['+', '/'], $payload)), true);
        
        // Check expiry
        if (isset($payload['exp']) && $payload['exp'] < time()) {
            return false;
        }

        return $payload;
    }
}

class AuthMiddleware {
    public static function verifyToken($request) {
        $headers = getallheaders();
        $authHeader = $headers['Authorization'] ?? $headers['authorization'] ?? null;

        if (!$authHeader || !str_starts_with($authHeader, 'Bearer ')) {
            self::unauthorized('Authorization token not provided');
            return false;
        }

        $token = substr($authHeader, 7);
        $payload = JWT::decode($token);

        if (!$payload) {
            self::unauthorized('Invalid or expired token');
            return false;
        }

        // Verify user still exists and is active
        $db = new Database();
        $conn = $db->getConnection();
        
        $stmt = $conn->prepare("SELECT id, username, email, role, status FROM users WHERE id = ? AND status = 'active'");
        $stmt->execute([$payload['user_id']]);
        $user = $stmt->fetch();

        if (!$user) {
            self::unauthorized('User not found or inactive');
            return false;
        }

        return $user;
    }

    public static function checkRole($user, $requiredRole) {
        $roleHierarchy = ['student' => 1, 'teacher' => 2, 'admin' => 3];
        $userLevel = $roleHierarchy[$user['role']] ?? 0;
        $requiredLevel = $roleHierarchy[$requiredRole] ?? 0;

        if ($userLevel < $requiredLevel) {
            self::forbidden('Insufficient permissions');
            return false;
        }

        return true;
    }

    public static function checkMadrasaAccess($user, $madrasa_id) {
        if ($user['role'] === 'admin') {
            return true; // Admin has access to all
        }

        $db = new Database();
        $conn = $db->getConnection();

        if ($user['role'] === 'teacher') {
            $stmt = $conn->prepare("SELECT id FROM teachers WHERE user_id = ? AND madrasa_id = ?");
        } else if ($user['role'] === 'student') {
            $stmt = $conn->prepare("SELECT id FROM students WHERE user_id = ? AND madrasa_id = ?");
        } else {
            return false;
        }

        $stmt->execute([$user['id'], $madrasa_id]);
        $result = $stmt->fetch();

        if (!$result) {
            self::forbidden('Access denied to this madrasa');
            return false;
        }

        return true;
    }

    public static function ok($data = null, $message = 'Success') {
        http_response_code(200);
        header('Content-Type: application/json');
        echo json_encode([
            'success' => true,
            'message' => $message,
            'data' => $data,
            'timestamp' => date('c')
        ]);
        exit;
    }

    public static function created($data = null, $message = 'Resource created successfully') {
        http_response_code(201);
        header('Content-Type: application/json');
        echo json_encode([
            'success' => true,
            'message' => $message,
            'data' => $data,
            'timestamp' => date('c')
        ]);
        exit;
    }

    public static function badRequest($message = 'Bad request') {
        http_response_code(400);
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => $message,
            'timestamp' => date('c')
        ]);
        exit;
    }

    public static function unauthorized($message = 'Unauthorized') {
        http_response_code(401);
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => $message,
            'timestamp' => date('c')
        ]);
        exit;
    }

    public static function forbidden($message = 'Forbidden') {
        http_response_code(403);
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => $message,
            'timestamp' => date('c')
        ]);
        exit;
    }

    public static function notFound($message = 'Resource not found') {
        http_response_code(404);
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => $message,
            'timestamp' => date('c')
        ]);
        exit;
    }

    public static function serverError($message = 'Internal server error') {
        http_response_code(500);
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => $message,
            'timestamp' => date('c')
        ]);
        exit;
    }
}

class RateLimiter {
    private static $limits = [];
    private static $requests = [];

    public static function check($identifier, $maxRequests = 60, $timeWindow = 60) {
        $now = time();
        $key = md5($identifier);

        if (!isset(self::$requests[$key])) {
            self::$requests[$key] = [];
        }

        // Remove old requests outside time window
        self::$requests[$key] = array_filter(self::$requests[$key], function($timestamp) use ($now, $timeWindow) {
            return $now - $timestamp < $timeWindow;
        });

        if (count(self::$requests[$key]) >= $maxRequests) {
            return false;
        }

        self::$requests[$key][] = $now;
        return true;
    }
}

// CORS Headers
function setCorsHeaders() {
    $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
    if (in_array($origin, Config::$allowed_origins) || in_array('*', Config::$allowed_origins)) {
        header("Access-Control-Allow-Origin: $origin");
    }
    
    header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
    header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
    header("Access-Control-Max-Age: 86400");
    
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(200);
        exit();
    }
}

// Request logging
function logRequest($userId = null, $action = '', $details = '') {
    $db = new Database();
    $conn = $db->getConnection();

    $madrasa_id = $_GET['madrasa_id'] ?? 1; // Default to first madrasa
    
    $stmt = $conn->prepare("INSERT INTO activity_logs (madrasa_id, user_id, action, ip_address, user_agent) VALUES (?, ?, ?, ?, ?)");
    $stmt->execute([
        $madrasa_id,
        $userId,
        $action,
        $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
    ]);
}
?>