/**
 * API Client for Madrasa Management System
 * Connects to PHP REST API Backend
 */

const API_BASE_URL = import.meta.env.VITE_API_URL || 'http://localhost/api';

class ApiClient {
  private baseURL: string;
  private token: string | null = null;

  constructor(baseURL: string = API_BASE_URL) {
    this.baseURL = baseURL;
    this.loadToken();
  }

  private loadToken() {
    this.token = localStorage.getItem('auth_token');
  }

  private saveToken(token: string) {
    this.token = token;
    localStorage.setItem('auth_token', token);
  }

  private removeToken() {
    this.token = null;
    localStorage.removeItem('auth_token');
  }

  private async request<T>(
    endpoint: string,
    options: RequestInit = {}
  ): Promise<T> {
    const url = `${this.baseURL}${endpoint}`;
    
    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
      'Accept': 'application/json',
    };

    if (this.token) {
      headers['Authorization'] = `Bearer ${this.token}`;
    }

    const config: RequestInit = {
      ...options,
      headers: {
        ...headers,
        ...options.headers,
      },
    };

    try {
      const response = await fetch(url, config);
      
      if (!response.ok) {
        const errorData = await response.json().catch(() => ({ 
          message: 'Network error' 
        }));
        
        throw new Error(errorData.message || `HTTP ${response.status}`);
      }

      const data = await response.json();
      return data;
    } catch (error) {
      if (error instanceof Error) {
        throw error;
      }
      throw new Error('Network request failed');
    }
  }

  // Authentication methods
  async login(username: string, password: string) {
    const response = await this.request<{
      success: boolean;
      message: string;
      data: {
        user: any;
        token: string;
        additional_data: any;
      };
    }>('/auth/login', {
      method: 'POST',
      body: JSON.stringify({ username, password }),
    });

    if (response.success) {
      this.saveToken(response.data.token);
    }
    
    return response;
  }

  async register(userData: {
    username: string;
    email: string;
    password: string;
    full_name: string;
    role: 'admin' | 'teacher' | 'student';
    madrasa_id?: number;
  }) {
    return this.request('/auth/register', {
      method: 'POST',
      body: JSON.stringify(userData),
    });
  }

  async getProfile() {
    return this.request('/auth/profile');
  }

  async updateProfile(userData: {
    full_name?: string;
    phone?: string;
    email?: string;
  }) {
    return this.request('/auth/profile', {
      method: 'PUT',
      body: JSON.stringify(userData),
    });
  }

  async changePassword(currentPassword: string, newPassword: string) {
    return this.request('/auth/change-password', {
      method: 'PUT',
      body: JSON.stringify({
        current_password: currentPassword,
        new_password: newPassword,
      }),
    });
  }

  async logout() {
    try {
      await this.request('/auth/logout', { method: 'POST' });
    } catch (error) {
      // Logout even if API call fails
      console.error('Logout API call failed:', error);
    } finally {
      this.removeToken();
    }
  }

  // Dashboard methods
  async getDashboard() {
    return this.request('/dashboard');
  }

  // Students methods
  async getStudents(params?: {
    page?: number;
    limit?: number;
    search?: string;
  }) {
    const queryParams = new URLSearchParams();
    if (params?.page) queryParams.append('page', params.page.toString());
    if (params?.limit) queryParams.append('limit', params.limit.toString());
    if (params?.search) queryParams.append('search', params.search);
    
    const query = queryParams.toString() ? `?${queryParams.toString()}` : '';
    return this.request(`/students${query}`);
  }

  async getStudent(id: number) {
    return this.request(`/students/${id}`);
  }

  async createStudent(studentData: {
    first_name: string;
    last_name: string;
    father_name: string;
    mother_name?: string;
    date_of_birth?: string;
    gender: 'male' | 'female';
    phone?: string;
    guardian_phone?: string;
    email: string;
    admission_date?: string;
  }) {
    return this.request('/students', {
      method: 'POST',
      body: JSON.stringify(studentData),
    });
  }

  async updateStudent(id: number, studentData: any) {
    return this.request(`/students/${id}`, {
      method: 'PUT',
      body: JSON.stringify(studentData),
    });
  }

  async deleteStudent(id: number) {
    return this.request(`/students/${id}`, {
      method: 'DELETE',
    });
  }

  // Teachers methods
  async getTeachers() {
    return this.request('/teachers');
  }

  async getTeacher(id: number) {
    return this.request(`/teachers/${id}`);
  }

  async createTeacher(teacherData: any) {
    return this.request('/teachers', {
      method: 'POST',
      body: JSON.stringify(teacherData),
    });
  }

  async updateTeacher(id: number, teacherData: any) {
    return this.request(`/teachers/${id}`, {
      method: 'PUT',
      body: JSON.stringify(teacherData),
    });
  }

  async deleteTeacher(id: number) {
    return this.request(`/teachers/${id}`, {
      method: 'DELETE',
    });
  }

  // Classes methods
  async getClasses() {
    return this.request('/classes');
  }

  async getClass(id: number) {
    return this.request(`/classes/${id}`);
  }

  async createClass(classData: any) {
    return this.request('/classes', {
      method: 'POST',
      body: JSON.stringify(classData),
    });
  }

  async updateClass(id: number, classData: any) {
    return this.request(`/classes/${id}`, {
      method: 'PUT',
      body: JSON.stringify(classData),
    });
  }

  async deleteClass(id: number) {
    return this.request(`/classes/${id}`, {
      method: 'DELETE',
    });
  }

  // Attendance methods
  async getAttendance(params?: {
    date?: string;
    class_id?: number;
  }) {
    const queryParams = new URLSearchParams();
    if (params?.date) queryParams.append('date', params.date);
    if (params?.class_id) queryParams.append('class_id', params.class_id.toString());
    
    const query = queryParams.toString() ? `?${queryParams.toString()}` : '';
    return this.request(`/attendance${query}`);
  }

  async markAttendance(attendanceData: {
    date: string;
    records: Array<{
      student_id: number;
      class_id?: number;
      status: 'present' | 'absent' | 'late' | 'excused';
      remarks?: string;
    }>;
  }) {
    return this.request('/attendance', {
      method: 'POST',
      body: JSON.stringify(attendanceData),
    });
  }

  // Fees methods
  async getFees(params?: {
    student_id?: number;
    status?: string;
  }) {
    const queryParams = new URLSearchParams();
    if (params?.student_id) queryParams.append('student_id', params.student_id.toString());
    if (params?.status) queryParams.append('status', params.status);
    
    const query = queryParams.toString() ? `?${queryParams.toString()}` : '';
    return this.request(`/fees${query}`);
  }

  async createFee(feeData: {
    student_id: number;
    fee_type: string;
    amount: number;
    due_date: string;
    status?: string;
  }) {
    return this.request('/fees', {
      method: 'POST',
      body: JSON.stringify(feeData),
    });
  }

  async updateFee(id: number, feeData: any) {
    return this.request(`/fees/${id}`, {
      method: 'PUT',
      body: JSON.stringify(feeData),
    });
  }

  // Income/Expenses methods
  async getIncome(params?: {
    start_date?: string;
    end_date?: string;
  }) {
    const queryParams = new URLSearchParams();
    if (params?.start_date) queryParams.append('start_date', params.start_date);
    if (params?.end_date) queryParams.append('end_date', params.end_date);
    
    const query = queryParams.toString() ? `?${queryParams.toString()}` : '';
    return this.request(`/income${query}`);
  }

  async createIncome(incomeData: any) {
    return this.request('/income', {
      method: 'POST',
      body: JSON.stringify(incomeData),
    });
  }

  async getExpenses(params?: {
    start_date?: string;
    end_date?: string;
  }) {
    const queryParams = new URLSearchParams();
    if (params?.start_date) queryParams.append('start_date', params.start_date);
    if (params?.end_date) queryParams.append('end_date', params.end_date);
    
    const query = queryParams.toString() ? `?${queryParams.toString()}` : '';
    return this.request(`/expenses${query}`);
  }

  async createExpense(expenseData: any) {
    return this.request('/expenses', {
      method: 'POST',
      body: JSON.stringify(expenseData),
    });
  }

  // Reports methods
  async getStudentReport(params?: {
    class_id?: number;
  }) {
    const queryParams = new URLSearchParams();
    if (params?.class_id) queryParams.append('class_id', params.class_id.toString());
    
    const query = queryParams.toString() ? `?${queryParams.toString()}` : '';
    return this.request(`/reports/students${query}`);
  }

  async getFinancialReport(params?: {
    start_date?: string;
    end_date?: string;
  }) {
    const queryParams = new URLSearchParams();
    if (params?.start_date) queryParams.append('start_date', params.start_date);
    if (params?.end_date) queryParams.append('end_date', params.end_date);
    
    const query = queryParams.toString() ? `?${queryParams.toString()}` : '';
    return this.request(`/reports/financial${query}`);
  }

  // File upload method
  async uploadFile(file: File, tableName: string, recordId: number) {
    const formData = new FormData();
    formData.append('file', file);
    formData.append('table_name', tableName);
    formData.append('record_id', recordId.toString());

    return this.request('/files/upload', {
      method: 'POST',
      body: formData,
      headers: {
        // Don't set Content-Type for FormData, browser will set it
        Authorization: this.token ? `Bearer ${this.token}` : '',
      },
    });
  }

  // Health check
  async healthCheck() {
    return this.request('/health');
  }

  // Utility methods
  isAuthenticated(): boolean {
    return !!this.token;
  }

  getToken(): string | null {
    return this.token;
  }
}

// Export singleton instance
export const apiClient = new ApiClient();
export default apiClient;