<?php
/**
 * User Registration Page
 * SaaS AI Application Generator
 * 
 * Author: MiniMax Agent
 * Date: 2025-11-07
 */

$errors = [];
$success = false;

// Handle form submission
if ($_POST) {
    $email = sanitizeInput($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';
    $name = sanitizeInput($_POST['name'] ?? '');
    $company = sanitizeInput($_POST['company'] ?? '');
    $acceptTerms = isset($_POST['accept_terms']);
    
    // Validation
    if (empty($email)) {
        $errors[] = 'Email is required';
    } elseif (!validateEmail($email)) {
        $errors[] = 'Please enter a valid email address';
    }
    
    if (empty($password)) {
        $errors[] = 'Password is required';
    } elseif (strlen($password) < 8) {
        $errors[] = 'Password must be at least 8 characters long';
    }
    
    if (empty($confirmPassword)) {
        $errors[] = 'Please confirm your password';
    } elseif ($password !== $confirmPassword) {
        $errors[] = 'Passwords do not match';
    }
    
    if (empty($name)) {
        $errors[] = 'Name is required';
    }
    
    if (!$acceptTerms) {
        $errors[] = 'You must accept the terms and conditions';
    }
    
    // Check if user already exists
    if (empty($errors)) {
        $existingUser = $db->fetch("SELECT id FROM users WHERE email = ?", [$email]);
        if ($existingUser) {
            $errors[] = 'An account with this email already exists';
        }
    }
    
    // Create user if no errors
    if (empty($errors)) {
        $result = register($email, $password, $name, $company);
        
        if ($result['success']) {
            $success = true;
            $successMessage = $result['message'];
        } else {
            $errors[] = $result['message'];
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sign Up - AI App Generator SaaS</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/auth.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="auth-page register">
    <div class="auth-container">
        <div class="auth-card">
            <div class="auth-header">
                <div class="auth-logo">
                    <i class="fas fa-magic"></i>
                    <span>AI App Generator</span>
                </div>
                <h1>Create Your Account</h1>
                <p>Start building amazing applications with AI power</p>
            </div>
            
            <?php if ($success): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    <?= htmlspecialchars($successMessage) ?>
                    <br>
                    <a href="?page=login" class="auth-link">Click here to sign in</a>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($errors)): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-triangle"></i>
                    <strong>Please fix the following errors:</strong>
                    <ul style="margin-top: 10px; margin-left: 20px;">
                        <?php foreach ($errors as $error): ?>
                            <li><?= htmlspecialchars($error) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <form method="post" id="registerForm" class="auth-form">
                <div class="form-row">
                    <div class="form-group">
                        <label for="name">Full Name</label>
                        <div class="input-group">
                            <i class="fas fa-user input-icon"></i>
                            <input type="text" id="name" name="name" required 
                                   placeholder="Enter your full name"
                                   value="<?= htmlspecialchars($_POST['name'] ?? '') ?>">
                        </div>
                        <div class="error-message" id="nameError"></div>
                    </div>
                    
                    <div class="form-group">
                        <label for="company">Company (Optional)</label>
                        <div class="input-group">
                            <i class="fas fa-building input-icon"></i>
                            <input type="text" id="company" name="company" 
                                   placeholder="Your company name"
                                   value="<?= htmlspecialchars($_POST['company'] ?? '') ?>">
                        </div>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="email">Email Address</label>
                    <div class="input-group">
                        <i class="fas fa-envelope input-icon"></i>
                        <input type="email" id="email" name="email" required 
                               placeholder="Enter your email"
                               value="<?= htmlspecialchars($_POST['email'] ?? '') ?>">
                    </div>
                    <div class="error-message" id="emailError"></div>
                </div>
                
                <div class="form-group">
                    <label for="password">Password</label>
                    <div class="input-group">
                        <i class="fas fa-lock input-icon"></i>
                        <input type="password" id="password" name="password" required 
                               placeholder="Create a strong password"
                               oninput="checkPasswordStrength(this.value)">
                        <button type="button" class="password-toggle" onclick="togglePassword('password')">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                    <div class="password-strength" id="passwordStrength" style="display: none;">
                        <div class="strength-meter">
                            <div class="strength-bar" id="strengthBar"></div>
                        </div>
                        <div class="strength-text" id="strengthText"></div>
                    </div>
                    <div class="error-message" id="passwordError"></div>
                </div>
                
                <div class="form-group">
                    <label for="confirm_password">Confirm Password</label>
                    <div class="input-group">
                        <i class="fas fa-lock input-icon"></i>
                        <input type="password" id="confirm_password" name="confirm_password" required 
                               placeholder="Confirm your password">
                        <button type="button" class="password-toggle" onclick="togglePassword('confirm_password')">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                    <div class="error-message" id="confirmPasswordError"></div>
                </div>
                
                <div class="terms-checkbox">
                    <label>
                        <input type="checkbox" name="accept_terms" value="1" 
                               <?= isset($_POST['accept_terms']) ? 'checked' : '' ?>>
                        <span class="checkbox-custom"></span>
                        I agree to the <a href="#terms" class="terms-link">Terms of Service</a> 
                        and <a href="#privacy" class="terms-link">Privacy Policy</a>
                    </label>
                </div>
                
                <button type="submit" class="btn-auth" id="registerBtn">
                    <i class="fas fa-user-plus"></i>
                    Create Account
                </button>
            </form>
            
            <div class="auth-footer">
                <p>Already have an account? 
                    <a href="?page=login" class="auth-link">Sign in here</a>
                </p>
            </div>
            
            <div class="auth-divider">
                <span>or</span>
            </div>
            
            <div class="social-login">
                <button class="btn-social btn-google" type="button" onclick="signUpWithGoogle()">
                    <i class="fab fa-google"></i>
                    Sign up with Google
                </button>
                <button class="btn-social btn-github" type="button" onclick="signUpWithGitHub()">
                    <i class="fab fa-github"></i>
                    Sign up with GitHub
                </button>
            </div>
            
            <div class="back-home">
                <a href="?page=home">
                    <i class="fas fa-arrow-left"></i>
                    Back to Home
                </a>
            </div>
        </div>
        
        <div class="auth-visual">
            <div class="auth-illustration">
                <div class="illustration-element element-1">
                    <i class="fas fa-rocket"></i>
                </div>
                <div class="illustration-element element-2">
                    <i class="fas fa-cogs"></i>
                </div>
                <div class="illustration-element element-3">
                    <i class="fas fa-chart-line"></i>
                </div>
                <div class="illustration-element element-4">
                    <i class="fas fa-globe"></i>
                </div>
                <div class="floating-text">
                    <h2>Start Building</h2>
                    <p>Your Future</p>
                </div>
            </div>
        </div>
    </div>
    
    <div class="auth-background">
        <div class="bg-shapes">
            <div class="shape shape-1"></div>
            <div class="shape shape-2"></div>
            <div class="shape shape-3"></div>
            <div class="shape shape-4"></div>
        </div>
    </div>
    
    <script>
        function togglePassword(fieldId) {
            const field = document.getElementById(fieldId);
            const icon = field.parentNode.querySelector('.password-toggle i');
            
            if (field.type === 'password') {
                field.type = 'text';
                icon.className = 'fas fa-eye-slash';
            } else {
                field.type = 'password';
                icon.className = 'fas fa-eye';
            }
        }
        
        function checkPasswordStrength(password) {
            const strengthContainer = document.getElementById('passwordStrength');
            const strengthBar = document.getElementById('strengthBar');
            const strengthText = document.getElementById('strengthText');
            
            if (!password) {
                strengthContainer.style.display = 'none';
                return;
            }
            
            strengthContainer.style.display = 'block';
            
            let score = 0;
            let feedback = [];
            
            // Length check
            if (password.length >= 8) score += 1;
            else feedback.push('at least 8 characters');
            
            // Lowercase check
            if (/[a-z]/.test(password)) score += 1;
            else feedback.push('lowercase letters');
            
            // Uppercase check
            if (/[A-Z]/.test(password)) score += 1;
            else feedback.push('uppercase letters');
            
            // Number check
            if (/[0-9]/.test(password)) score += 1;
            else feedback.push('numbers');
            
            // Special character check
            if (/[^A-Za-z0-9]/.test(password)) score += 1;
            else feedback.push('special characters');
            
            // Update UI based on score
            let level, text, className;
            
            if (score < 2) {
                level = 'weak';
                text = 'Weak password';
                className = 'strength-weak';
            } else if (score < 4) {
                level = 'medium';
                text = 'Medium password';
                className = 'strength-medium';
            } else {
                level = 'strong';
                text = 'Strong password';
                className = 'strength-strong';
            }
            
            strengthBar.className = 'strength-bar ' + className;
            strengthText.textContent = text;
        }
        
        function signUpWithGoogle() {
            // Implement Google OAuth
            showNotification('Google sign up coming soon!', 'info');
        }
        
        function signUpWithGitHub() {
            // Implement GitHub OAuth
            showNotification('GitHub sign up coming soon!', 'info');
        }
        
        // Form validation
        document.getElementById('registerForm').addEventListener('submit', function(e) {
            const name = document.getElementById('name').value;
            const email = document.getElementById('email').value;
            const password = document.getElementById('password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            const termsAccepted = document.querySelector('input[name="accept_terms"]').checked;
            
            // Clear previous errors
            ['nameError', 'emailError', 'passwordError', 'confirmPasswordError'].forEach(id => {
                document.getElementById(id).textContent = '';
            });
            
            let isValid = true;
            
            // Name validation
            if (!name.trim()) {
                document.getElementById('nameError').textContent = 'Name is required';
                isValid = false;
            }
            
            // Email validation
            if (!email.trim()) {
                document.getElementById('emailError').textContent = 'Email is required';
                isValid = false;
            } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
                document.getElementById('emailError').textContent = 'Please enter a valid email';
                isValid = false;
            }
            
            // Password validation
            if (!password) {
                document.getElementById('passwordError').textContent = 'Password is required';
                isValid = false;
            } else if (password.length < 8) {
                document.getElementById('passwordError').textContent = 'Password must be at least 8 characters';
                isValid = false;
            }
            
            // Confirm password validation
            if (!confirmPassword) {
                document.getElementById('confirmPasswordError').textContent = 'Please confirm your password';
                isValid = false;
            } else if (password !== confirmPassword) {
                document.getElementById('confirmPasswordError').textContent = 'Passwords do not match';
                isValid = false;
            }
            
            // Terms validation
            if (!termsAccepted) {
                showNotification('Please accept the terms and conditions', 'error');
                isValid = false;
            }
            
            if (!isValid) {
                e.preventDefault();
            } else {
                // Show loading state
                const btn = document.getElementById('registerBtn');
                btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creating Account...';
                btn.disabled = true;
            }
        });
        
        // Auto-focus first field
        document.getElementById('name').focus();
    </script>
</body>
</html>