<?php
/**
 * Core Functions
 * SaaS AI Application Generator
 * 
 * Author: MiniMax Agent
 * Date: 2025-11-07
 */

// Global database instance
global $db;

/**
 * Security Functions
 */
function generateCSRFToken() {
    if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
        $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
    }
    return $_SESSION[CSRF_TOKEN_NAME];
}

function validateCSRFToken($token) {
    return isset($_SESSION[CSRF_TOKEN_NAME]) && hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
}

function sanitizeInput($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

function validatePassword($password) {
    return strlen($password) >= PASSWORD_MIN_LENGTH;
}

function generateSecurePassword($length = 12) {
    return bin2hex(random_bytes($length));
}

function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Session Management
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

function getCurrentUser() {
    if (!isLoggedIn()) {
        return null;
    }
    
    global $db;
    $user = $db->fetch("SELECT * FROM users WHERE id = ? AND status = 'active'", [$_SESSION['user_id']]);
    return $user ?: null;
}

function requireAuth() {
    if (!isLoggedIn()) {
        header('Location: ?page=login');
        exit();
    }
}

function requireAdmin() {
    requireAuth();
    $user = getCurrentUser();
    if ($user['role'] !== 'admin') {
        header('Location: ?page=dashboard');
        exit();
    }
}

/**
 * User Authentication
 */
function login($email, $password) {
    global $db;
    
    $user = $db->fetch("SELECT * FROM users WHERE email = ? AND status = 'active'", [$email]);
    
    if (!$user) {
        return ['success' => false, 'message' => 'Invalid email or password'];
    }
    
    if (!verifyPassword($password, $user['password'])) {
        return ['success' => false, 'message' => 'Invalid email or password'];
    }
    
    // Update last login
    $db->update('users', 
        ['last_login' => date('Y-m-d H:i:s')], 
        'id = ?', 
        [$user['id']]
    );
    
    // Set session
    $_SESSION['user_id'] = $user['id'];
    $_SESSION['user_email'] = $user['email'];
    $_SESSION['user_role'] = $user['role'];
    
    return ['success' => true, 'user' => $user];
}

function logout() {
    session_destroy();
    session_start();
    session_regenerate_id(true);
}

function register($email, $password, $name, $company = null) {
    global $db;
    
    // Check if user already exists
    if ($db->exists('users', 'email = ?', [$email])) {
        return ['success' => false, 'message' => 'User already exists with this email'];
    }
    
    // Validate inputs
    if (!validateEmail($email)) {
        return ['success' => false, 'message' => 'Invalid email address'];
    }
    
    if (!validatePassword($password)) {
        return ['success' => false, 'message' => 'Password must be at least ' . PASSWORD_MIN_LENGTH . ' characters'];
    }
    
    // Create user
    $userData = [
        'email' => $email,
        'password' => hashPassword($password),
        'name' => $name,
        'company' => $company,
        'role' => 'user',
        'status' => 'active',
        'plan_id' => 'free',
        'created_at' => date('Y-m-d H:i:s'),
        'email_verified' => false,
        'verification_token' => bin2hex(random_bytes(32))
    ];
    
    $userId = $db->insert('users', $userData);
    
    if ($userId) {
        // Send welcome email
        sendWelcomeEmail($email, $name, $userData['verification_token']);
        
        return [
            'success' => true, 
            'message' => 'Registration successful. Please check your email for verification.',
            'user_id' => $userId
        ];
    }
    
    return ['success' => false, 'message' => 'Registration failed. Please try again.'];
}

/**
 * Subscription Management
 */
function getUserPlan($userId) {
    global $db;
    
    $user = $db->fetch("SELECT plan_id, subscription_status FROM users WHERE id = ?", [$userId]);
    if (!$user) {
        return null;
    }
    
    $planDetails = PLANS[$user['plan_id']] ?? PLANS['free'];
    $planDetails['subscription_status'] = $user['subscription_status'] ?? 'inactive';
    
    return $planDetails;
}

function checkUsageLimit($userId, $type) {
    global $db;
    
    $user = $db->fetch("SELECT plan_id FROM users WHERE id = ?", [$userId]);
    if (!$user) {
        return false;
    }
    
    $plan = PLANS[$user['plan_id']] ?? PLANS['free'];
    $limits = $plan['limits'] ?? [];
    
    // Get current month usage
    $currentMonth = date('Y-m');
    
    switch ($type) {
        case 'projects':
            $currentUsage = $db->count('projects', 'user_id = ? AND DATE_FORMAT(created_at, "%Y-%m") = ?', [$userId, $currentMonth]);
            $limit = $limits['projects_per_month'] ?? 0;
            return $limit === -1 || $currentUsage < $limit;
            
        case 'ai_calls':
            $currentUsage = $db->count('ai_generations', 'user_id = ? AND DATE_FORMAT(created_at, "%Y-%m") = ?', [$userId, $currentMonth]);
            $limit = $limits['ai_calls_per_month'] ?? 0;
            return $limit === -1 || $currentUsage < $limit;
            
        case 'storage':
            $currentUsage = $db->fetch("SELECT COALESCE(SUM(file_size), 0) as total_size FROM user_files WHERE user_id = ?", [$userId]);
            $currentSize = $currentUsage['total_size'] / (1024 * 1024); // Convert to MB
            $limit = $limits['storage_mb'] ?? 0;
            return $currentSize < $limit;
    }
    
    return true;
}

/**
 * AI Generation Functions
 */
function generateApplication($userId, $prompt, $options = []) {
    global $db;
    
    // Check usage limits
    if (!checkUsageLimit($userId, 'projects')) {
        return ['success' => false, 'message' => 'Monthly project limit reached. Please upgrade your plan.'];
    }
    
    // Check AI calls limit
    if (!checkUsageLimit($userId, 'ai_calls')) {
        return ['success' => false, 'message' => 'Monthly AI generation limit reached. Please upgrade your plan.'];
    }
    
    // Create generation record
    $generationData = [
        'user_id' => $userId,
        'prompt' => $prompt,
        'options' => json_encode($options),
        'status' => 'generating',
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    $generationId = $db->insert('ai_generations', $generationData);
    
    if (!$generationId) {
        return ['success' => false, 'message' => 'Failed to create generation record'];
    }
    
    try {
        // Get user's plan for AI provider selection
        $userPlan = getUserPlan($userId);
        $provider = $options['provider'] ?? 'openai';
        
        // Generate application
        $result = callAIProvider($provider, $prompt, $options);
        
        if ($result['success']) {
            // Update generation status
            $db->update('ai_generations', 
                ['status' => 'completed', 'result' => json_encode($result['data'])], 
                'id = ?', 
                [$generationId]
            );
            
            // Create project
            $projectData = [
                'user_id' => $userId,
                'name' => $options['project_name'] ?? 'AI Generated Application',
                'description' => $prompt,
                'generation_id' => $generationId,
                'files' => json_encode($result['data']['files']),
                'tech_stack' => $result['data']['tech_stack'] ?? 'HTML/CSS/JS',
                'status' => 'active',
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $projectId = $db->insert('projects', $projectData);
            
            // Send completion notification
            sendProjectCompletionEmail($userId, $projectId);
            
            return [
                'success' => true, 
                'message' => 'Application generated successfully!',
                'project_id' => $projectId,
                'generation_id' => $generationId
            ];
        } else {
            // Update generation status
            $db->update('ai_generations', 
                ['status' => 'failed', 'error_message' => $result['message']], 
                'id = ?', 
                [$generationId]
            );
            
            return $result;
        }
        
    } catch (Exception $e) {
        // Update generation status
        $db->update('ai_generations', 
            ['status' => 'failed', 'error_message' => $e->getMessage()], 
            'id = ?', 
            [$generationId]
        );
        
        return ['success' => false, 'message' => 'Generation failed: ' . $e->getMessage()];
    }
}

function callAIProvider($provider, $prompt, $options) {
    switch ($provider) {
        case 'openai':
            return callOpenAI($prompt, $options);
        case 'claude':
            return callClaude($prompt, $options);
        case 'gemini':
            return callGemini($prompt, $options);
        default:
            return ['success' => false, 'message' => 'Invalid AI provider'];
    }
}

function callOpenAI($prompt, $options) {
    $apiKey = OPENAI_API_KEY;
    if (empty($apiKey) || $apiKey === 'your_openai_api_key') {
        return ['success' => false, 'message' => 'OpenAI API key not configured'];
    }
    
    $data = [
        'model' => 'gpt-4',
        'messages' => [
            [
                'role' => 'system',
                'content' => 'You are an expert full-stack developer. Generate complete, production-ready web applications based on user requirements.'
            ],
            [
                'role' => 'user',
                'content' => $prompt
            ]
        ],
        'max_tokens' => 4000,
        'temperature' => 0.7
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/chat/completions');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Bearer ' . $apiKey
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        return ['success' => false, 'message' => 'OpenAI API request failed'];
    }
    
    $result = json_decode($response, true);
    $content = $result['choices'][0]['message']['content'] ?? '';
    
    // Parse the generated code (you'll need to implement proper parsing)
    return [
        'success' => true,
        'data' => [
            'files' => parseGeneratedCode($content),
            'tech_stack' => 'HTML/CSS/JavaScript/PHP',
            'description' => $content
        ]
    ];
}

function parseGeneratedCode($content) {
    // This is a simplified version. In a real implementation, 
    // you'd have more sophisticated parsing and validation
    return [
        'index.html' => $content,
        'style.css' => '/* Generated styles will be added here */',
        'script.js' => '// Generated JavaScript will be added here',
        'README.md' => '# Generated Application'
    ];
}

function callClaude($prompt, $options) {
    $apiKey = CLAUDE_API_KEY;
    if (empty($apiKey) || $apiKey === 'your_claude_api_key') {
        return ['success' => false, 'message' => 'Claude API key not configured'];
    }
    
    // Implement Claude API call here
    return ['success' => false, 'message' => 'Claude integration not yet implemented'];
}

function callGemini($prompt, $options) {
    $apiKey = GEMINI_API_KEY;
    if (empty($apiKey) || $apiKey === 'your_gemini_api_key') {
        return ['success' => false, 'message' => 'Gemini API key not configured'];
    }
    
    // Implement Gemini API call here
    return ['success' => false, 'message' => 'Gemini integration not yet implemented'];
}

/**
 * Email Functions
 */
function sendWelcomeEmail($email, $name, $token) {
    // Implement email sending logic here
    // You can use PHPMailer or similar
    return true;
}

function sendProjectCompletionEmail($userId, $projectId) {
    global $db;
    
    $user = $db->fetch("SELECT email, name FROM users WHERE id = ?", [$userId]);
    if (!$user) {
        return false;
    }
    
    // Implement email sending logic
    return true;
}

/**
 * Utility Functions
 */
function formatBytes($size, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    
    for ($i = 0; $size > 1024 && $i < count($units) - 1; $i++) {
        $size /= 1024;
    }
    
    return round($size, $precision) . ' ' . $units[$i];
}

function timeAgo($datetime) {
    $time = time() - strtotime($datetime);
    
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . ' minutes ago';
    if ($time < 86400) return floor($time/3600) . ' hours ago';
    if ($time < 2592000) return floor($time/86400) . ' days ago';
    if ($time < 31536000) return floor($time/2592000) . ' months ago';
    return floor($time/31536000) . ' years ago';
}

function redirect($url) {
    header('Location: ' . $url);
    exit();
}

function jsonResponse($data, $status = 200) {
    http_response_code($status);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit();
}

/**
 * Error Handling
 */
function handleError($message) {
    if (APP_DEBUG) {
        return ['error' => $message, 'debug' => true];
    } else {
        return ['error' => 'An error occurred. Please try again.'];
    }
}
?>